import type { IPosition } from 'monaco-editor';

import { Operator } from '../../api/models/Operator';
import { ExpressionNode, KeyNode, OperatorNode, QueryNode, ValueNode, ValuesNode } from '../grammar/nodes';
import { BaseVisitor, parse } from '../grammar/parser';
import { QuotedValue } from '../grammar/tokens';

import {
   isInsideOrRightAfterLocation,
   isRightAfterLocation,
   Location,
   locationToRange,
   positionToRange,
   reduceToCurrent,
} from './locationHelpers';
import { SuggestCase, SuggestRequest } from './models';

interface VisitorParams {
   fieldName?: string;
   position: IPosition;
}

class Visitor extends BaseVisitor {
   constructor() {
      super();

      this.validateVisitor();
   }

   public query(ctx: QueryNode['children'], { position }: VisitorParams): SuggestRequest {
      const currentExpression = ctx.expression?.reduce(reduceToCurrent(position));

      if (!currentExpression) {
         return {
            case: SuggestCase.Empty,
            fieldName: undefined,
            query: '',
            range: positionToRange(position),
            type: 'key',
         };
      }

      if (!isInsideOrRightAfterLocation(currentExpression.location as Location, position)) {
         return {
            case: SuggestCase.AfterSemicolon,
            fieldName: undefined,
            query: '',
            range: positionToRange(position),
            type: 'key',
         };
      }

      return this.visit(currentExpression, { position });
   }

   public expression(ctx: ExpressionNode['children'], { position }: { position: IPosition }): SuggestRequest {
      const emptyRequest: SuggestRequest = {
         case: SuggestCase.NoSuggestions,
         fieldName: undefined,
         query: '',
         range: positionToRange(position),
         type: 'empty',
      };

      const keyNode = ctx.key[0];
      const fieldName: string = this.visit(ctx.key);

      // Key
      const keyLocation = keyNode.location as Location;
      if (isInsideOrRightAfterLocation(keyLocation, position)) {
         const suggestCase = isRightAfterLocation(keyLocation, position) ? SuggestCase.AfterKey : SuggestCase.InTheKey;
         return {
            case: suggestCase,
            fieldName: undefined,
            query: fieldName.substring(0, position.column - keyLocation.startColumn),
            range: locationToRange(keyLocation),
            type: 'key',
         };
      }

      // Hypothetical situation
      if (!ctx.operator) {
         return emptyRequest;
      }

      // Operator
      const operatorNode = ctx.operator[0];
      const operatorLocation = operatorNode.location as Location;
      if (isInsideOrRightAfterLocation(operatorLocation, position)) {
         const range = locationToRange(operatorLocation);
         return {
            case: isRightAfterLocation(operatorLocation, position)
               ? SuggestCase.AfterOperator
               : SuggestCase.InTheOperator,
            fieldName,
            query: '',
            range: { ...range, startColumn: position.column, endColumn: position.column },
            type: 'value',
         };
      }

      if (!ctx.values) {
         return emptyRequest;
      }

      // Values
      const valuesNode = ctx.values[0];
      const valuesLocation = valuesNode.location as Location;
      if (isInsideOrRightAfterLocation(valuesLocation, position)) {
         return this.visit(valuesNode, { position, fieldName });
      }

      return emptyRequest;
   }

   public key(ctx: KeyNode['children']): string {
      return ctx.Value[0].image;
   }

   public operator(ctx: OperatorNode['children']): Operator | null {
      if (ctx.Grep) {
         return Operator.Grep;
      }

      if (ctx.NotGrep) {
         return Operator.NotGrep;
      }

      if (ctx.NotEqual) {
         return Operator.NotEqual;
      }

      return Operator.Equal;
   }

   public values(ctx: ValuesNode['children'], { position, fieldName }: VisitorParams): SuggestRequest {
      const currentValue = ctx.value?.reduce(reduceToCurrent(position));
      if (currentValue && isRightAfterLocation(currentValue.location as Location, position)) {
         return this.visit(currentValue, { position, fieldName });
      }

      return {
         case: SuggestCase.AfterComma,
         fieldName,
         query: '',
         range: positionToRange(position),
         type: 'value',
      };
   }

   public value(ctx: ValueNode['children'], { position, fieldName }: VisitorParams): SuggestRequest {
      const valueToken = [ctx.Value?.[0], ctx.QuotedValue?.[0]].filter(Boolean).reduce(reduceToCurrent(position));
      const valueLocation = valueToken as Location;

      return {
         case: valueToken.tokenType.name === QuotedValue.name ? SuggestCase.InQuotedValue : SuggestCase.InsideValue,
         fieldName,
         query: valueToken.image,
         range: locationToRange(valueLocation),
         type: 'value',
      };
   }
}

const visitor = new Visitor();

export function getSuggestRequest(query: string, position: IPosition): SuggestRequest {
   return visitor.visit(parse(query), { position });
}
