import { Meta, Story } from '@storybook/react';
import { DevJson, formatNumber, sortHandler } from '@yandex-infracloud-ui/libs';
import { differenceInCalendarYears } from 'date-fns';
import * as React from 'react';
import { useCallback, useState } from 'react';
import { Observable, of } from 'rxjs';

import { DateCell } from './cells/DateCell/DateCell';
import { LimitFilter } from './filters/LimitFilter/LimitFilter';
import { TextFilter } from './filters/TextFilter/TextFilter';
import { CellProps, Order, PlainObject } from './models';
import { columnPresets } from './models/ColumnPreset';
import { SmartTable } from './SmartTable';

interface User {
   birthday: string;
   login: string;
   name: string;
}

const AgeCell: React.FC<CellProps> = ({ rowEntry }) => {
   const birthday = new Date(rowEntry.birthday);
   const age = differenceInCalendarYears(new Date(), birthday);

   return <>{formatNumber(age)}</>;
};

const ST = SmartTable;

const items: User[] = [
   { name: 'ivan', login: 'bIvan', birthday: '1995-06-21' },
   { name: 'petr', login: 'petrI', birthday: '2000-06-21' },
   { name: 'sergey', login: 'grizzle', birthday: '1993-01-21' },
   { name: 'artem', login: 'tema', birthday: '1990-06-21' },
];

const Template: Story<{ initialQueryString: string }> = ({ initialQueryString }) => {
   const [queryString, setQueryString] = useState(initialQueryString);

   const getItems = useCallback(
      (params: PlainObject): Observable<{ items: User[]; nextToken: string | undefined }> =>
         of({
            items: [
               ...(params.order
                  ? items.sort((a, b) =>
                       params.order === Order.DESC
                          ? sortHandler(b.birthday, a.birthday)
                          : sortHandler(a.birthday, b.birthday),
                    )
                  : items),
            ].filter(item =>
               params.query ? item.name.includes(params.query) || item.login.includes(params.query) : true,
            ),
            nextToken: undefined,
         }),
      [],
   );

   return (
      <>
         <DevJson open={true}>{queryString}</DevJson>

         <SmartTable
            queryString={queryString}
            requiredFilters={[]}
            onQueryStringUpdate={setQueryString}
            getItems={getItems}
         >
            <ST.Filters>
               <ST.Filter
                  name={'query'}
                  defaultValue={''}
                  component={TextFilter}
                  internalProps={{ placeholder: 'Query' }}
               />
               <ST.ColumnPresets />
               <ST.Filter name={'limit'} defaultValue={50} component={LimitFilter} />
               <ST.FilterButtons />
            </ST.Filters>

            <ST.Table idColumns={['login']} orderColumn={'birthday'}>
               <ST.Column name={'login'} header={'User login'} presets={columnPresets.default} />
               <ST.Column name={'name'} presets={columnPresets.default} />
               <ST.Column name={'age'} component={AgeCell} presets={columnPresets.details} />
               <ST.Column name={'birthday'} header={'Birthday'} component={DateCell} presets={columnPresets.ignored} />
            </ST.Table>
         </SmartTable>
      </>
   );
};

export const Default = Template.bind({});
Default.args = { initialQueryString: '' };

export default {
   title: 'smart-table/SmartTable',
} as Meta;
