import { useDismounted } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import {
   PropsWithChildren,
   ReactNode,
   useCallback,
   useEffect,
   useImperativeHandle,
   useMemo,
   useRef,
   useState,
} from 'react';
import { finalize, Observable, takeUntil } from 'rxjs';

import { Column } from './components/Column/Column';
import { ColumnPresets } from './components/ColumnPresets/ColumnPresets';
import { Filter, FilterWrapperProps } from './components/Filter/Filter';
import { FilterButtons } from './components/FilterButtons/FilterButtons';
import { Filters } from './components/Filters/Filters';
import { Table } from './components/Table/Table';
import { findChildren } from './helpers/findChildren';
import { useEffectSkipFirst } from './hooks/useEffectSkipFirst';
import { SmartTableContextValue, SmartTableProvider } from './hooks/useSmartTableContext';
import { FilterParamsList, FiltersHook, useSmartTableFilters } from './hooks/useSmartTableFilters';
import { GetItemsResponse, Order, PlainObject } from './models';
import { ColumnPreset } from './models/ColumnPreset';

function getFiltersParams(children: ReactNode): FilterParamsList {
   const filtersNodes = findChildren(children, Filter);
   const result: FilterParamsList = {};

   for (const filterNode of filtersNodes) {
      const props = (filterNode as any).props as FilterWrapperProps<any>;
      const { name, defaultValue, getValueFromUrlParams, setUrlParamsFromValue } = props;

      result[name] = {
         defaultValue,
         getValueFromUrlParams,
         setUrlParamsFromValue,
      };
   }

   return result;
}

export interface SmartTableRef {
   getFiltersHook(): FiltersHook;
}

interface Props {
   queryString: string;
   requiredFilters: string[];

   onQueryStringUpdate(urlParams: string): void;

   getItems(filters: PlainObject, token: string | undefined): Observable<GetItemsResponse>;
}

const SmartTableImpl = React.forwardRef<SmartTableRef, PropsWithChildren<Props>>(
   ({ queryString, requiredFilters, onQueryStringUpdate, getItems, children }, ref) => {
      const dismounted = useDismounted();
      const [isLoading, setIsLoading] = useState(false);
      const [resp, setResp] = useState<GetItemsResponse>();

      const filterParamListRef = useRef(getFiltersParams(children));
      const filtersHook = useSmartTableFilters(queryString, filterParamListRef.current);

      const { filters, getUrlParams, setFilter, submitFilters } = filtersHook;
      const { order } = filters;

      const skipHandleLoad = useMemo(() => requiredFilters.some(v => !filters[v]), [filters, requiredFilters]);

      useEffectSkipFirst(() => {
         onQueryStringUpdate(getUrlParams());
      }, [onQueryStringUpdate, filters]);

      const toggleOrder = useCallback(() => {
         setFilter('order', order === Order.ASC ? Order.DESC : Order.ASC);
         submitFilters();
      }, [order, setFilter, submitFilters]);

      const handleLoad = useCallback(() => {
         if (!skipHandleLoad) {
            setIsLoading(true);

            getItems(filters, undefined)
               .pipe(
                  finalize(() => setIsLoading(false)),
                  takeUntil(dismounted),
               )
               .subscribe(setResp);
         }
      }, [dismounted, filters, skipHandleLoad, getItems]);

      const handleLoadMore = useCallback(() => {
         setIsLoading(true);

         getItems(filters, resp?.nextToken)
            .pipe(
               finalize(() => setIsLoading(false)),
               takeUntil(dismounted),
            )
            .subscribe(newResp => {
               setResp(r => ({
                  items: [...(r?.items ?? []), ...newResp.items],
                  nextToken: newResp.nextToken,
               }));
            });
      }, [dismounted, filters, getItems, resp?.nextToken]);

      const [columnPreset, setColumnPreset] = useState(ColumnPreset.Default);

      const contextValue: SmartTableContextValue = useMemo(
         () => ({
            filtersHook,

            handleLoadMore,
            isLoading,
            response: resp,

            columnPreset,
            setColumnPreset,

            toggleOrder,
         }),
         [columnPreset, filtersHook, handleLoadMore, isLoading, resp, toggleOrder],
      );

      // Initial data load and reload on filters changes
      useEffect(() => {
         handleLoad();
      }, [handleLoad]);

      useImperativeHandle(
         ref,
         () => ({
            getFiltersHook() {
               return filtersHook;
            },
         }),
         [filtersHook],
      );

      return (
         <SmartTableProvider value={contextValue}>
            {/*
     <DevJson open={false} summary={'filters'}>
               {{ filters }}
            </DevJson>
            */}
            {children}
         </SmartTableProvider>
      );
   },
);

SmartTableImpl.displayName = 'SmartTable';

export const SmartTable = Object.assign(SmartTableImpl, {
   Filters,
   Filter,
   FilterButtons,
   ColumnPresets,
   Table,
   Column,
});
