import { Button, Loader, TableColumnConfig, TableSettingsData, withTableSettings } from '@yandex-cloud/uikit';
import { classNames } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { ReactElement, ReactNode, useCallback, useMemo, useState } from 'react';

import { StringCell } from '../../cells/StringCell/StringCell';
import { findChildren } from '../../helpers/findChildren';
import { useSmartTableContext } from '../../hooks/useSmartTableContext';
import { PlainObject } from '../../models';
import { ColumnPreset } from '../../models/ColumnPreset';
import { Column, ColumnProps } from '../Column/Column';

import { ExpandableTable } from './hoc/ExpandableTable';
import { withOrderColumn } from './hoc/withOrderColumn';
import classes from './Table.module.css';

const MyTable = withTableSettings(withOrderColumn(ExpandableTable));

function isColumn(child: ReactNode): boolean {
   return (child as any).type === Column;
}

export function getTableRowId(item: PlainObject, idColumns: string[]): string {
   return idColumns.map(id => item[id]).join('-');
}

export interface TableProps {
   className?: string;
   idColumns: string[];
   orderColumn: string;

   getDefaultOpenedRowIdList?(items: PlainObject[]): string[];
   renderSecondRow?(item: PlainObject): ReactNode;
}

export const Table: React.FC<TableProps> = React.memo(
   ({ className, idColumns, orderColumn, getDefaultOpenedRowIdList, renderSecondRow, children: rawChildren }) => {
      const {
         filtersHook: {
            actual,
            filters: { order },
         },
         columnPreset,
         handleLoadMore,
         isLoading,
         response,
         toggleOrder,
      } = useSmartTableContext();

      // TODO: DEPLOY-5824
      const { columnsConfig, columnPresets } = useMemo(() => {
         const columns: TableColumnConfig<any>[] = [];
         const cpresets: Record<ColumnPreset, TableSettingsData> = {
            [ColumnPreset.Default]: [],
            [ColumnPreset.Details]: [],
         };

         findChildren(rawChildren, Column).forEach(child => {
            if (!child || !isColumn(child)) {
               return;
            }
            const { name, header, component, presets } = (child as ReactElement).props as ColumnProps;

            columns.push({
               id: name,
               name: header,
               template(item: any) {
                  const Cell = component ?? StringCell;

                  return <Cell rowEntry={item} columnName={name} />;
               },
            });

            cpresets[ColumnPreset.Default].push({
               id: name,
               isSelected: presets.includes(ColumnPreset.Default),
            });
            cpresets[ColumnPreset.Details].push({
               id: name,
               isSelected: presets.includes(ColumnPreset.Details),
            });
         });

         return { columnsConfig: columns, columnPresets: cpresets };
      }, [rawChildren]);

      const [settings, setSettings] = useState<TableSettingsData>(columnPresets[columnPreset]);

      const handleUpdateSettings = useCallback((s: TableSettingsData) => {
         setSettings(s);

         return Promise.resolve();
      }, []);

      // TODO: DEPLOY-5824
      // сбрасываются настройки колонок при любых изменениях фильтров,
      // стоит переделать в этом месте, пока просто закомментирую,
      // лучше всего - уметь сохранять выбранные настройки пользователя где-нибудь в localStorage
      // useEffect(() => {
      //    setSettings(columnPresets[columnPreset]);
      // }, [columnPreset, columnPresets]);

      const getRowClassNames = useCallback(() => [classes.firstRow], []);
      const getRowId = useCallback(item => getTableRowId(item, idColumns), [idColumns]);

      return (
         <div className={classNames(classes.wrapper, className)}>
            {response ? (
               <>
                  <MyTable
                     className={classNames(classes.table, {
                        [classes.isNotActual]: !actual,
                        [classes.isLoading]: isLoading,
                     })}
                     columnWithOrder={orderColumn}
                     columns={columnsConfig}
                     data={response.items}
                     emptyMessage={'No data'}
                     expandedRowClassName={classes.expanded}
                     getRowClassNames={getRowClassNames}
                     getRowId={getRowId}
                     onOrderToggle={toggleOrder}
                     order={order}
                     renderSecondRow={renderSecondRow}
                     secondRowClassName={classes.secondRow}
                     settings={settings}
                     updateSettings={handleUpdateSettings}
                     verticalAlign={'top'}
                     getDefaultOpenedRowIdList={getDefaultOpenedRowIdList}
                  />

                  {response.nextToken || isLoading ? (
                     <Button
                        className={classes.loadMore}
                        loading={isLoading}
                        onClick={handleLoadMore}
                        disabled={isLoading}
                     >
                        Load more...
                     </Button>
                  ) : null}
               </>
            ) : (
               <Loader className={classes.loader} size={'l'} />
            )}
         </div>
      );
   },
);

Table.displayName = 'SmartTable.Table';
