import { faChevronDown } from '@fortawesome/pro-regular-svg-icons/faChevronDown';
import { faChevronUp } from '@fortawesome/pro-regular-svg-icons/faChevronUp';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Table, TableProps } from '@yandex-cloud/uikit';
import { classNames, isEqual, toggleSetItem } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { ReactNode } from 'react';

import { PlainObject } from '../../../models';

import classes from './ExpandableTable.module.css';

type Item = PlainObject;

interface Props extends TableProps<Item> {
   expandedRowClassName?: string;
   secondRowClassName?: string;

   getDefaultOpenedRowIdList?(items: Item[]): string[];
   renderSecondRow?(item: Item, index: number): ReactNode;
}

interface State {
   openedRows: Set<string>;
}

export class ExpandableTable extends React.Component<Props, State> {
   /**
    * original table component
    */
   private table: Table | null = null;

   constructor(props: Props) {
      super(props);

      this.state = {
         openedRows: new Set(props.getDefaultOpenedRowIdList ? props.getDefaultOpenedRowIdList(props.data) : []),
      };

      this.table = this.initTable(props);
   }

   componentDidUpdate(prevProps: Props) {
      if (!isEqual(prevProps, this.props)) {
         this.table = this.initTable(this.props);
         this.forceUpdate();
      }
   }

   initTable(props: Props) {
      const { columns, renderSecondRow, secondRowClassName, expandedRowClassName } = props;

      const getId = (item: Item) => Table.getRowId(props, item);
      const isExpanded = (item: Item) => this.state.openedRows.has(getId(item));

      const table = new Table({
         ...props,
         className: classNames(classes.table, props.className),
         columns: [
            {
               id: 'expanded',
               sticky: 'left',
               name: '',
               width: 5,
               template: (item: Item) => (
                  <FontAwesomeIcon className={classes.icon} icon={isExpanded(item) ? faChevronUp : faChevronDown} />
               ),
            },
            ...columns,
         ],
         // Just for adding expandedRowClassName
         getRowClassNames: (item: Item, index: number): string[] => [
            classes.firstRow,
            ...(props.getRowClassNames?.(item, index) ?? []),
            classNames(
               expandedRowClassName && {
                  [expandedRowClassName]: isExpanded(item),
                  [classes.expanded]: isExpanded(item),
               },
            ),
         ],
         onRowClick: item => {
            this.setState(s => ({ openedRows: toggleSetItem(s.openedRows, getId(item)) }));
         },
      });

      // @ts-ignore
      const originalRenderRow = table.renderRow.bind(table);

      // @ts-ignore
      table.renderRow = (item: Item, rowIndex: number) => {
         const secondRow =
            renderSecondRow && isExpanded(item) ? (
               <tr className={classNames(classes.secondRow, secondRowClassName)}>
                  <td className={'yc-table__cell'} />
                  <td className={'yc-table__cell'} colSpan={table.props.columns.length - 1}>
                     {renderSecondRow(item, rowIndex)}
                  </td>
               </tr>
            ) : null;

         return (
            <React.Fragment key={getId(item)}>
               {originalRenderRow(item, rowIndex)}
               {secondRow}
            </React.Fragment>
         );
      };

      return table;
   }

   render() {
      return this.table?.render() ?? null;
   }
}
