import { fromQuery, isEqual, toQuery } from '@yandex-infracloud-ui/libs';
import { useCallback, useMemo, useReducer } from 'react';

import { FilterWrapperProps } from '../components/Filter/Filter';
import { plainToNested } from '../helpers/nestedToPlain';
import { PlainObject } from '../models';

type FilterParams<T> = Omit<FilterWrapperProps<T>, 'component' | 'name'>;

export type FilterParamsList = Record<string, FilterParams<any>>;

function getFiltersFromUrlParams(qs: string, filterParamList: FilterParamsList): PlainObject {
   const parsedQuery = plainToNested(fromQuery(qs));

   const result: PlainObject = {};

   for (const [name, filterParams] of Object.entries(filterParamList)) {
      const { defaultValue, getValueFromUrlParams } = filterParams;

      if (getValueFromUrlParams) {
         result[name] = getValueFromUrlParams(parsedQuery, name);
      } else if (parsedQuery[name]) {
         const isNumber = typeof defaultValue === 'number';
         result[name] = isNumber ? parseInt(parsedQuery[name] as string, 10) : parsedQuery[name];
      } else {
         result[name] = defaultValue;
      }
   }

   return result;
}

function getUrlParamsFromFilters(filterParamList: FilterParamsList, filters: PlainObject): string {
   const result: PlainObject = {};
   for (const [name, { defaultValue, setUrlParamsFromValue }] of Object.entries(filterParamList)) {
      const value = filters[name];
      if (setUrlParamsFromValue) {
         Object.assign(result, setUrlParamsFromValue(value));
      } else if (!isEqual(value, defaultValue)) {
         result[name] = value;
      }
   }

   return toQuery(result);
}

export interface FiltersHook {
   actual: boolean;
   filters: PlainObject;

   clearFilters(): void;

   getFilter<T>(name: string): T;

   getCurrentFilters(): PlainObject;

   getUrlParams(): string;

   resetFilters(): void;

   setFilter<T>(name: string, v: T): void;

   submitFilters(): void;
}

// region Reducer
interface FiltersState {
   defaultValue: PlainObject;
   externalValue: PlainObject;
   internalValue: PlainObject;
}

const emptyState: FiltersState = {
   defaultValue: {},
   externalValue: {},
   internalValue: {},
};

enum ActionType {
   Set = 'Set',
   Submit = 'Submit',
   Reset = 'Reset',
   Clear = 'Clear',
}

interface SetAction {
   name: string;
   type: ActionType.Set;
   value: any;
}

interface SubmitAction {
   type: ActionType.Submit;
}

interface ResetAction {
   type: ActionType.Reset;
}

interface ClearAction {
   type: ActionType.Clear;
}

type Action = SetAction | SubmitAction | ResetAction | ClearAction;

function init(queryString: string, paramsList: FilterParamsList): FiltersState {
   const initialValue = getFiltersFromUrlParams(queryString, paramsList);

   return {
      defaultValue: getFiltersFromUrlParams('', paramsList),
      externalValue: initialValue,
      internalValue: initialValue,
   };
}

function reducer(state: FiltersState, action: Action): FiltersState {
   switch (action.type) {
      case ActionType.Set: {
         return {
            ...state,
            internalValue: {
               ...state.internalValue,
               [action.name]: action.value,
            },
         };
      }
      case ActionType.Submit: {
         return {
            ...state,
            externalValue: state.internalValue,
         };
      }
      case ActionType.Reset: {
         return {
            ...state,
            internalValue: state.externalValue,
         };
      }
      case ActionType.Clear: {
         return {
            ...state,
            internalValue: state.defaultValue,
            externalValue: state.defaultValue,
         };
      }
      default: {
         return state;
      }
   }
}

// endregion

export function useSmartTableFilters(queryString: string, paramsList: FilterParamsList): FiltersHook {
   const [state, dispatch] = useReducer(reducer, emptyState, () => init(queryString, paramsList));

   const clearFilters = useCallback(() => {
      dispatch({ type: ActionType.Clear });
   }, []);

   const getCurrentFilters = useCallback(() => state.internalValue, [state.internalValue]);

   const getFilter = useCallback(<T>(name: string): T => state.internalValue[name], [state.internalValue]);

   const getUrlParams = useCallback(() => getUrlParamsFromFilters(paramsList, state.externalValue), [
      paramsList,
      state.externalValue,
   ]);

   const setFilter = useCallback(<T>(name: string, value: T) => dispatch({ type: ActionType.Set, name, value }), []);

   const resetFilters = useCallback(() => dispatch({ type: ActionType.Reset }), []);

   const submitFilters = useCallback(() => dispatch({ type: ActionType.Submit }), []);

   return useMemo(
      () => ({
         actual: isEqual(state.externalValue, state.internalValue),
         filters: state.externalValue,

         clearFilters,
         getCurrentFilters,
         getFilter,
         getUrlParams,
         resetFilters,
         setFilter,
         submitFilters,
      }),
      [
         clearFilters,
         getCurrentFilters,
         getFilter,
         getUrlParams,
         resetFilters,
         setFilter,
         state.externalValue,
         state.internalValue,
         submitFilters,
      ],
   );
}
