import React from 'react';

import { RequestState } from '../../../redux/slices/network/model';
import { StatusInfo, StatusMap, StatusState } from '../../lib';

import classes from './RequestMap.module.css';

type Props = {
   statuses: (Omit<StatusInfo, 'status'> & { requestState: RequestState | null })[];
   showCount?: boolean;
};

const requestStatuses = {
   [RequestState.OK]: StatusState.Ok,
   [RequestState.ERROR]: StatusState.Error,
   [RequestState.PENDING]: StatusState.Progress,
};

const stateStyle: Partial<Record<StatusState, string>> = {
   [StatusState.Ok]: classes.ok,
   [StatusState.Error]: classes.error,
   [StatusState.Progress]: classes.progress,
   [StatusState.Inactive]: classes.inactive,
};

export const RequestMap: React.FC<Props> = React.memo(({ statuses, showCount = false }) => {
   const requestCounts = statuses.reduce((counts, info) => {
      const { requestState } = info;
      const state = requestState ? requestStatuses[requestState] : StatusState.Inactive;
      counts.set(state, (counts.get(state) ?? 0) + 1);
      return counts;
   }, new Map<StatusState, number>());
   return (
      <>
         {showCount && (
            <div className={classes.counts}>
               {[...requestCounts.entries()].map(([state, count]) => (
                  <span key={state} title={`${count} of ${state}`} className={stateStyle[state]}>
                     {count}
                  </span>
               ))}
            </div>
         )}
         <StatusMap
            statuses={statuses.map(info => {
               const { requestState, ...data } = info;
               return {
                  ...data,
                  status: requestState ? requestStatuses[requestState] : StatusState.Inactive,
               };
            })}
         />
      </>
   );
});

RequestMap.displayName = 'RequestMap';
