import { classNames, modalService, TIMES_IN_MS } from '@yandex-infracloud-ui/libs';
import { Spin } from 'lego-on-react';
import React, { useCallback } from 'react';

import { LegoButton } from '../../../_lego';
import { RequestTimeoutItem } from '../../../models/api';
import { SecondTimer } from '../SecondTimer/SecondTimer';
import { TimeoutModal } from '../TimeoutModal/TimeoutModal';

import classes from './TimeoutToast.module.css';

interface Props {
   requests: RequestTimeoutItem[];
}

export const TimeoutToast: React.FC<Props> = React.memo(({ requests }) => {
   const now = Date.now();

   const onDetailsClick = useCallback(() => {
      modalService.open(TimeoutModal, { requests }).subscribe();
   }, [requests]);

   const onReloadPage = useCallback(() => {
      window.location.reload();
   }, []);

   const requestGroupsMap = new Map<number, RequestTimeoutItem[]>();
   let longTime = false;
   for (const request of requests) {
      const diffSeconds = Math.ceil((now - request.timestamp) / TIMES_IN_MS.Second);
      if (!longTime && diffSeconds > 60) {
         longTime = true;
      }
      if (!requestGroupsMap.has(diffSeconds)) {
         requestGroupsMap.set(diffSeconds, []);
      }
      requestGroupsMap.get(diffSeconds)!.push(request);
   }

   return (
      <div className={classes.plate}>
         {Array.from(requestGroupsMap.keys()).map(diffSeconds => (
            <div className={classes.timestampGroup} key={diffSeconds}>
               <div className={classes.timestamp}>
                  <SecondTimer timestamp={requestGroupsMap.get(diffSeconds)![0].timestamp} />s ago
               </div>
               <div>
                  {requestGroupsMap.get(diffSeconds)!.map(request => (
                     <div
                        className={classNames(classes.url, { [classes.aborted]: request.aborted })}
                        title={request.url}
                        key={`${request.timestamp}-${request.url}`}
                     >
                        {!request.aborted && (
                           <div className={classes.spin}>
                              <Spin size={'xxs'} progress />
                           </div>
                        )}
                        {request.url}
                     </div>
                  ))}
               </div>
            </div>
         ))}
         <div className={classes.actions}>
            <LegoButton onClick={onDetailsClick} size={'xs'}>
               Details
            </LegoButton>
            {requests.some(e => e.aborted) && (
               <LegoButton onClick={onReloadPage} size={'xs'}>
                  Reload page
               </LegoButton>
            )}
         </div>
      </div>
   );
});

TimeoutToast.displayName = 'TimeoutToast';
