import { Position } from '@yandex-infracloud-ui/query-input';

import { matchLocation } from './matchLocation';
import { ExpressionNode, OperatorNode, QueryNode, ValueNode, ValuesNode } from './nodes';
import { VisitorConstructor } from './QueryParser';

export class CurrentTokenInterpreter extends VisitorConstructor {
   constructor() {
      super();
      this.validateVisitor();
   }

   public query(ctx: QueryNode, position: Position) {
      if (!ctx.expression) {
         return undefined;
      }

      const currentExpression = ctx.expression.reduce((expr1, expr2) => {
         if (expr2.location!.startLine! > position.lineNumber || expr2.location!.startColumn! > position.column) {
            return expr1;
         }

         return expr2;
      });

      return this.visit(currentExpression, position);
   }

   protected expression(ctx: ExpressionNode, position: Position) {
      const keyToken = ctx.Value[0];
      const key = keyToken.image;

      if (ctx.ExpressionSeparator) {
         const separator = ctx.ExpressionSeparator[0]!;

         if (position.lineNumber >= separator.startLine! && position.column > separator.endColumn!) {
            return undefined;
         }
      }

      if (matchLocation(keyToken, position)) {
         return {
            key,
            type: 'KEY',
            value: key,

            endColumn: keyToken.endColumn,
            endLine: keyToken.endLine,
            startColumn: keyToken.startColumn,
            startLine: keyToken.startLine,
         };
      }

      const token = this.visit(ctx.values, position);

      if (!token) {
         const operatorToken = this.visit(ctx.operator, position);

         if (operatorToken) {
            return {
               ...operatorToken,
               key,
               type: 'VALUE',
               value: '',
            };
         }

         return {
            key,
            type: 'VALUE',
            value: '',
         };
      }

      return { key, type: 'VALUE', ...token };
   }

   protected operator(ctx: OperatorNode, position: Position) {
      const tokensArray = ctx.Equal || ctx.NotEqual;
      const token = tokensArray![0];

      if (!matchLocation(token, position)) {
         return undefined;
      }

      return {
         endColumn: token.endColumn,
         endLine: token.endLine,
         startColumn: token.endColumn,
         startLine: token.endLine,
      };
   }

   protected value(ctx: ValueNode) {
      if (ctx.QuotedValue) {
         return undefined;
      }
      const token = ctx.Value![0];

      return {
         value: token.image,

         endColumn: token.endColumn,
         endLine: token.endLine,
         startColumn: token.startColumn,
         startLine: token.startLine,
      };
   }

   protected values(ctx: ValuesNode, position: Position) {
      if (!ctx.value) {
         return undefined;
      }

      const currentValue = ctx.value.find(item => matchLocation(item.location!, position));

      if (!currentValue) {
         return undefined;
      }

      return this.visit(currentValue, position);
   }
}
