import { Position, Provider } from '@yandex-infracloud-ui/query-input';
import { Lexer } from 'chevrotain';

import throttle from 'lodash-es/throttle';
import { editor as monacoEditor, languages, Position as monacoPosition } from 'monaco-editor/esm/vs/editor/editor.api';

import { TIMEOUTS } from '../../../../old-code/constants/api';
import api from '../../../../old-code/services/api';

import { CurrentTokenInterpreter } from './CurrentTokenInterpreter';
import { QueryInterpreter } from './QueryInterpreter';
import { queryParser } from './QueryParser';
import { RequestInterpreter } from './RequestInterpreter';
import { tokens } from './tokens';

const QueryLexer = new Lexer(tokens);

export const suggestionKeys = [
   'log_level',
   'host',
   'pod',
   'box',
   'workload',
   'container_id',
   'logger_name',
   'pod_transient_fqdn',
   'pod_persistent_fqdn',
   'node_fqdn',
   'thread_name',
];

export const ignoreKeys = new Set(['message', 'stack_trace', 'context', 'log_level_int']);

export const aliases = new Map([
   ['container_id_list', 'container_list'],
   ['message_list', 'search_pattern_list'],
]);

const serializer = new QueryInterpreter();
const requester = new RequestInterpreter();
const tokenator = new CurrentTokenInterpreter();

interface PlainObject {
   [key: string]: string;
}

export class StageLogsProvider implements Provider {
   static isContextSuggest(query: string, position: Position) {
      return false;
   }

   private cache = new Map<string, Promise<languages.CompletionItem[]>>();

   // eslint-disable-next-line react/static-property-placement
   private readonly context: PlainObject;

   private isFullSearch = false;

   private loadMoreCommandId: string | null = null;

   constructor(context: PlainObject) {
      this.context = context;
      this.onSuggestValues = throttle(this.onSuggestValues, 1000) as any;
   }

   public initializeEditor(editor: monacoEditor.IStandaloneCodeEditor) {
      this.loadMoreCommandId = editor.addCommand(0, () => {
         this.isFullSearch = true;
         setTimeout(() => editor.trigger('suggest', 'editor.action.triggerSuggest', ''));
      });
   }

   public async onSuggest(
      model: monacoEditor.ITextModel,
      position: monacoPosition,
   ): Promise<languages.CompletionItem[]> {
      const inputString = model.getValue();

      const suggestHash = `${inputString}_${position.column}_${position.lineNumber}_${this.isFullSearch}`;
      const cachedResult = this.cache.get(suggestHash);

      if (cachedResult) {
         return cachedResult;
      }

      queryParser.input = QueryLexer.tokenize(inputString).tokens;
      const query = queryParser.query();
      const token = tokenator.visit(query, position);
      const ast = serializer.visit(query) || {};

      if (!inputString || !query || !token || token.type === 'KEY') {
         const currentToken = queryParser.getToken(position);

         if (currentToken && currentToken.image.startsWith('context.')) {
            return this.onSuggestContextKeys(currentToken, position);
         }

         return this.onSuggestKeys(currentToken, position, ast);
      }

      if (token) {
         if (token.type === 'VALUE') {
            if (token.key.startsWith('context.') || ignoreKeys.has(token.key)) {
               return [];
            }
            const suggestValuesResult = this.onSuggestValues(token, position, query);
            this.cache.clear();
            this.cache.set(suggestHash, suggestValuesResult);

            // eslint-disable-next-line no-return-await
            return await suggestValuesResult;
         }
      }

      return [] as languages.CompletionItem[];
   }

   private async onSuggestContextKeys(token: any, position: Position, ast = {}) {
      const keyPrefix = token.image.slice(8, position.column - token.startColumn);

      const { candidates } = (await api.request({
         action: 'GetQueryContextKeys',
         service: 'logs',
         timeout: TIMEOUTS.LOGS,

         data: {
            request: { ...this.context, ...ast },
            keyPrefix,
         },

         // TODO need typings for remove
         params: undefined,
         query: undefined,
      })) as { candidates: string[] };

      // bug DEPLOY-5864
      return candidates.map(item => {
         item = `context.${item}=`;

         return {
            insertText: item,
            kind: languages.CompletionItemKind.Text,
            label: item,
            range: {
               endColumn: token.endColumn + 2,
               endLineNumber: token.endLine,
               startColumn: token.startColumn,
               startLineNumber: token.startLine,
            },
         };
      }) as languages.CompletionItem[];
   }

   // TODO may be need partial suggest by position
   private onSuggestKeys(token: any, position: Position, ast: any) {
      const items = suggestionKeys
         .concat(['message', 'stack_trace'])
         .filter(key => !ast.hasOwnProperty(key))
         .map(item => ({
            insertText: `${item}=`,
            kind: languages.CompletionItemKind.Text,
            label: `${item}=`,
         }));

      items.push({
         insertText: 'context.',
         kind: languages.CompletionItemKind.Text,
         label: 'context.',
      });

      return items as languages.CompletionItem[];
   }

   private async onSuggestValues(token: any, position: Position, query: any) {
      const { key, value } = token;

      if (suggestionKeys.indexOf(key) === -1) {
         return [];
      }

      const valuePrefix = value.slice(0, position.column - token.startColumn);
      const request = requester.visit(query) || {};
      delete request[`${key}_list`];

      for (const [alias, original] of aliases.entries()) {
         if (request.hasOwnProperty(alias)) {
            request[original] = request[alias];
            delete request[alias];
         }
      }

      const fullSearch = this.isFullSearch;
      this.isFullSearch = false;

      const { candidates } = (await api.request({
         action: 'GetSearchQuerySuggests',
         service: 'logs',
         timeout: TIMEOUTS.LOGS,

         data: {
            fullSearch,
            keyType: key.toUpperCase(),
            request: { ...this.context, ...request },
            valuePrefix,
         },

         // TODO need typings for remove
         params: undefined,
         query: undefined,
      })) as { candidates: string[] };

      const result: any[] = candidates.map(item => {
         if (!/^[^,=;\s]*$/.test(item)) {
            // reserved symbols must be wrapped
            item = `"${item}"`;
         }

         let insertText = item;
         let label = item;

         if (item === '') {
            label = '<EMPTY>';
            insertText = '""';
         }

         insertText += '; ';

         return {
            insertText,
            filterText: item,
            kind: languages.CompletionItemKind.Text,
            label,
            range:
               token.startColumn !== token.endColumn
                  ? {
                       endColumn: token.endColumn + 2,
                       endLineNumber: token.endLine,
                       startColumn: token.startColumn,
                       startLineNumber: token.startLine,
                    }
                  : undefined,
         };
      });

      if (!fullSearch) {
         result.push({
            filterText: ` ${valuePrefix}`,
            insertText: '',
            label: 'load more...',
            preselect: false,
            sortText: 'z',

            range: {
               endColumn: position.column,
               endLineNumber: position.lineNumber,
               startColumn: position.column,
               startLineNumber: position.lineNumber,
            },

            command: {
               id: this.loadMoreCommandId,
               title: 'loadMore',
            },
         });
      }

      return result as languages.CompletionItem[];
   }
}
