import { IToken, Lexer } from 'chevrotain';
import { Comma, Equal, ExpressionSeparator, NotEqual, QuotedValue, Value, WhiteSpace } from './tokens';

const toValue = (token: IToken) => token.image;
const toTokenType = (token: IToken) => token.tokenType;

describe('tokens', () => {
   it('Value', () => {
      const parsedString = new Lexer([Value, WhiteSpace]).tokenize('a a.b a.b.');
      expect(parsedString.tokens.map(toValue)).toEqual(['a', 'a.b', 'a.b.']);
      expect(parsedString.tokens.map(toTokenType)).toEqual([Value, Value, Value]);
      expect(parsedString.errors).toEqual([]);
   });

   it('Quoted Value', () => {
      const parsedString = new Lexer([QuotedValue, WhiteSpace]).tokenize('"a" "a b"');
      expect(parsedString.tokens.map(toValue)).toEqual(['"a"', '"a b"']);
      expect(parsedString.tokens.map(toTokenType)).toEqual([QuotedValue, QuotedValue]);
      expect(parsedString.errors).toEqual([]);
   });

   it('Equals', () => {
      const parsedString = new Lexer([Equal, NotEqual, WhiteSpace]).tokenize('= !=');
      expect(parsedString.tokens.map(toValue)).toEqual(['=', '!=']);
      expect(parsedString.tokens.map(toTokenType)).toEqual([Equal, NotEqual]);
      expect(parsedString.errors).toEqual([]);
   });

   it('Whitespace', () => {
      const parsedString = new Lexer([Value, WhiteSpace]).tokenize('a b \n c \n\n\n d   \t');
      expect(parsedString.tokens.map(toValue)).toEqual(['a', 'b', 'c', 'd']);
      expect(parsedString.tokens.map(toTokenType)).toEqual([Value, Value, Value, Value]);
      expect(parsedString.errors).toEqual([]);
   });

   it('Token invalid', () => {
      const parsedString = new Lexer([Value, WhiteSpace]).tokenize('a, b');
      expect(parsedString.errors.length).toBe(1);
   });

   it('Expression', () => {
      const parsedString = new Lexer([Value, QuotedValue, Comma, ExpressionSeparator, NotEqual, WhiteSpace]).tokenize(
         'a.b!=value1, "value 2";',
      );
      expect(parsedString.tokens.map(toValue)).toEqual(['a.b', '!=', 'value1', ',', '"value 2"', ';']);
      expect(parsedString.tokens.map(toTokenType)).toEqual([
         Value,
         NotEqual,
         Value,
         Comma,
         QuotedValue,
         ExpressionSeparator,
      ]);
   });
});
