import { EnumSwitcher } from '@yandex-infracloud-ui/libs';

import { isBefore } from 'date-fns';
import * as H from 'history';
import React, { ReactElement, ReactNode, useCallback, useEffect, useMemo, useRef, useState } from 'react';
import { Prompt, useRouteMatch } from 'react-router';

import { DeepNonNullable } from '../../models/api';

import { entityLevel, Stage, StageConverter } from '../../models/ui';
import { StageConfirmations } from '../../models/ui/Confirmations';
import { SecretsReloadContext } from '../../models/ui/secrets';
import { ResourcesContext } from '../../models/ui/stage/resources';
import { TProject, TStage } from '../../proto-typings';
import { useConfig } from '../../services';
import { parseYpTimestamp, useBeforeUnload } from '../../utils';
import { FormChangeListener, HugeForm } from '../huge-form';
import { StageDiffView, StagePatcherOptions, StageQuota } from './components';
import { StageHugeYamlEditor } from './components/StageHugeYamlEditor/StageHugeYamlEditor'; // circular dependency (jest 'allStages' test)

import { formPlates } from './components/plates';
import { configs, LevelEmptyParamsMap } from './configs';
import { useIsApprovalRequired } from './formHooks';
import { isEqualStageForms, StageHugeFormEditView, StageHugeFormMode, StageHugeFormShowPage } from './models';

import {
   useStageCreateFormActions,
   useStageEditFormActions,
   useStageForm,
   useStageFormAbc,
   useStageFormProject,
   useStageFormQuota,
} from './hooks';
import { useEditView } from './useEditView';

import classes from './StageHugeForm.module.css';

const unloadConfirmation = 'You have unsaved changes. Are you sure you want to close the tab and lose them?';

const editViewOptions = [
   {
      value: StageHugeFormEditView.Form,
      content: 'form',
   },
   {
      value: StageHugeFormEditView.Raw,
      content: (
         <span>
            yaml <span style={{ color: 'red' }}>β</span>
         </span>
      ),
   },
];

export interface RenderArgs {
   content: ReactNode;
   contentEditViewSwitcher: ReactNode;
   hasUnsavedChanges: boolean;
   showPage: StageHugeFormShowPage;
   hasSelectedUpdateConfirmations: boolean;
   isSubmitting: boolean;
   isDisabledActions: boolean;

   update(): void;

   revert(): void;

   cancel(): void;
}

interface Props {
   description?: string;
   mode: StageHugeFormMode;
   latestRawStage?: TStage;
   rawStage: TStage;
   stage?: Stage;

   children(renderProps: RenderArgs): ReactElement;

   onCancel?(): void;
}

export const StageHugeForm: React.FC<Props> = React.memo(
   ({ children, description = '', latestRawStage, mode, onCancel, rawStage, stage }) => {
      const initialValue = useMemo(() => stage ?? StageConverter.fromApi(rawStage), [stage, rawStage]);

      const isNew = mode === StageHugeFormMode.New;

      const stageId = initialValue.id;
      const match = useRouteMatch();
      const rootRoutePath = useCallback(() => match.url, [match.url]);

      const rootFormId = rootRoutePath(); // TODO remove/refactor

      const {
         stageConfirmations,
         editedValue,
         handleCancelUpdate,
         handleRevertChanges,
         handleUpdate,
         hasUnsavedChanges,
         hugeFormRef,
         isSubmitting,
         reloadSecretUsages,
         cloneDuInSecrets,
         showPage,
      } = useStageForm({ rootFormId, mode, value: initialValue, rawStage });

      const {
         editView,
         formValue,
         rawValue,
         validationInfo,
         hasYamlErrors,
         parseYamlError,
         onChangeValidate,
         toogleEditView,
         yamlText,
         setYamlText,
         getCurrentForms,
         getCurrentStageValue,
         revertValues,
      } = useEditView({
         initialValue,
         rawStage,
         hugeFormRef,
         latestRawStage,
         rootRoutePath,
         isNew,
      });

      const config = useConfig()!;

      const showReloadWarning = config.showReloadStageWarning() && hasUnsavedChanges;

      const isApprovalRequired = useIsApprovalRequired(stageId, isNew);

      // for new stage
      const { stageReset: newStageReset, handleDeploy: newStageHandleDeploy } = useStageCreateFormActions({
         rootFormId,
         editedValue,
      });

      // for existing stage
      const {
         stageReset: existStageReset,
         handleDeploy: existStageHandleDeploy,
         handleDeployDraft,
      } = useStageEditFormActions({
         editedValue,
         latestRawStage,
         mode,
         rawStage: rawValue,
         rootFormId,
         stageId,
      });

      const stageReset = isNew ? newStageReset : existStageReset;

      const handleDeploy = isNew ? newStageHandleDeploy : existStageHandleDeploy;

      const handleReviewChanges = useCallback(() => {
         const currentStageValue = getCurrentStageValue();
         const forms = getCurrentForms();
         handleUpdate(currentStageValue, forms, editView === StageHugeFormEditView.Raw);
      }, [editView, getCurrentForms, getCurrentStageValue, handleUpdate]);

      const revert = useCallback(() => {
         stageReset();

         handleRevertChanges();

         // сброс для пересчёта зависимостей
         revertValues();
      }, [stageReset, handleRevertChanges, revertValues]);

      const cancel = useCallback(() => {
         if (showPage === StageHugeFormShowPage.Diff && mode === StageHugeFormMode.ApplyAsIs) {
            onCancel?.();
            return;
         }

         if (showPage === StageHugeFormShowPage.Diff || showPage === StageHugeFormShowPage.Update) {
            handleCancelUpdate();
            return;
         }

         revert();
         onCancel?.();
      }, [handleCancelUpdate, mode, showPage, onCancel, revert]);

      const currentEmptyParamsRef = useRef<LevelEmptyParamsMap>(new Map());

      const getProjectData = useCallback(
         (p: DeepNonNullable<TProject>) => [p.spec.account_id, p.spec.monitoring_project],
         [],
      );
      const { project, projectChangeListener, projectId } = useStageFormProject(initialValue, getProjectData);

      const { abc, initialAbc, abcChangeListener } = useStageFormAbc(initialValue, project);

      useEffect(() => {
         const params = currentEmptyParamsRef.current;
         params.set(entityLevel.deployUnit, { projectId: projectId ?? undefined });
         params.set(entityLevel.workload, { projectId: projectId ?? undefined });
      }, [projectId]);

      const hugeFormPayload = useMemo(() => {
         const stageDate = rawStage?.status?.spec_timestamp
            ? parseYpTimestamp(rawStage?.status?.spec_timestamp)
            : new Date();

         return {
            // FIXME временная мера https://st.yandex-team.ru/DEPLOY-3287
            showDiskIsolationField: isBefore(stageDate, new Date('2020-09-16T16:05:00+0300')),
            monitoringProject: project?.spec?.monitoring_project,
            projectId,
         };
      }, [rawStage?.status?.spec_timestamp, project?.spec?.monitoring_project, projectId]);

      const {
         quotaChangeListeners,
         oldResourceValues,
         newResourceValues,
         usedClusters,
         currentQuota,
         resourceContextValue,
      } = useStageFormQuota({
         isNew,
         abc,
         initialAbc,
         stage: initialValue,
      });

      const formChangeListeners: FormChangeListener[] = useMemo(
         () => [...quotaChangeListeners, projectChangeListener, abcChangeListener],
         [projectChangeListener, quotaChangeListeners, abcChangeListener],
      );

      const handleBeforeChangeLocation = useCallback(
         (location: H.Location) => {
            // Внутри структуры стейджа навигацию разрешаем
            if (location.pathname.startsWith(rootFormId)) {
               return true;
            }

            return unloadConfirmation as string;
         },
         [rootFormId],
      );

      useBeforeUnload(showReloadWarning, unloadConfirmation);

      // Явное затирание данных при уходе с формы (мы же все равно предупреждаем)
      useEffect(() => stageReset, [stageReset]);

      const isEditable = mode === StageHugeFormMode.Edit || mode === StageHugeFormMode.Apply;
      const warning =
         hugeFormPayload.showDiskIsolationField && isEditable ? (
            <div className={classes.warning}>{formPlates['DEPLOY-3287']}</div>
         ) : null;

      // для сохранения значений уже выбранных чекбоксов:
      // 1) по дефолту выключаем все чекбоксы, чтобы избежать незапланированных пользователем обновлений спеки
      // при случайном двойном клике по кнопке update при редактировании стейджа,
      // 2) обнуляем их при возвращении к редактированию конфига, т.к. пользователь может накликать галочек,
      // а потом вернуться и вручную эти поля отредактировать (и эти поля уже не нужно будет обновлять)
      const [selectedDuConfirmations, setSelectedDuConfirmations] = useState<StageConfirmations>({});

      const hasSelectedUpdateConfirmations = useMemo(
         () => Object.values(selectedDuConfirmations).some(v => v.size > 0),
         [selectedDuConfirmations],
      );

      const isDisabledActions = hasYamlErrors || parseYamlError !== null;
      const contentEditViewSwitcher = (
         <EnumSwitcher
            className={classes.toggleRawSwitcher}
            name={'toggleRaw'}
            value={editView}
            onChange={toogleEditView}
            disabled={showPage !== StageHugeFormShowPage.Edit || isDisabledActions}
            options={editViewOptions}
         />
      );

      const showForm = editView === StageHugeFormEditView.Form;
      const readonly = mode === StageHugeFormMode.View;

      const content = (
         <SecretsReloadContext.Provider value={reloadSecretUsages}>
            <ResourcesContext.Provider value={resourceContextValue}>
               <Prompt when={showReloadWarning} message={handleBeforeChangeLocation} />

               {/* hidden используется, чтобы не переинициализировать <HugeForm/> */}
               <div hidden={showPage !== StageHugeFormShowPage.Edit}>
                  {warning}

                  <div hidden={!showForm}>
                     <HugeForm
                        disabled={showPage !== StageHugeFormShowPage.Edit || isSubmitting}
                        formChangeListeners={formChangeListeners}
                        isNew={isNew}
                        levelConfigs={configs}
                        onFormRevert={reloadSecretUsages}
                        onNodeClone={cloneDuInSecrets}
                        onNodeListChange={reloadSecretUsages}
                        payload={hugeFormPayload}
                        readonly={readonly}
                        ref={hugeFormRef}
                        rootRoutePath={rootRoutePath}
                        value={formValue}
                        initialValue={initialValue}
                        currentEmptyParams={currentEmptyParamsRef.current}
                        isEqualForms={isEqualStageForms}
                     />
                  </div>

                  {abc && showForm && (
                     <StageQuota
                        tree={currentQuota ?? null}
                        clusters={usedClusters}
                        editableResources={{
                           old: oldResourceValues,
                           new: newResourceValues,
                        }}
                        abc={abc}
                        initialAbc={initialAbc}
                        viewMode={readonly}
                     />
                  )}

                  {editView === StageHugeFormEditView.Raw && (
                     <StageHugeYamlEditor
                        value={yamlText}
                        readonly={readonly}
                        onUpdate={setYamlText}
                        onChangeValidate={onChangeValidate}
                        validationInfo={validationInfo}
                     />
                  )}
               </div>

               {showPage === StageHugeFormShowPage.Update && (
                  <StagePatcherOptions
                     {...stageConfirmations}
                     onChange={v => {
                        setSelectedDuConfirmations(v);
                     }}
                  />
               )}

               {showPage === StageHugeFormShowPage.Diff && (
                  <StageDiffView
                     description={description}
                     duConfirmations={selectedDuConfirmations}
                     latestRawStage={latestRawStage ?? rawStage}
                     isApprovalRequired={isApprovalRequired}
                     mode={mode}
                     onCancel={handleCancelUpdate}
                     onDeploy={handleDeploy}
                     onDeployDraft={handleDeployDraft}
                     specValue={rawValue}
                     value={editedValue}
                  />
               )}
            </ResourcesContext.Provider>
         </SecretsReloadContext.Provider>
      );

      return children({
         cancel,
         content,
         contentEditViewSwitcher,
         hasUnsavedChanges,
         isSubmitting,
         isDisabledActions,
         revert,
         showPage,
         update: handleReviewChanges,
         hasSelectedUpdateConfirmations,
      });
   },
);

StageHugeForm.displayName = 'StageHugeForm';
