import { HelpPopover } from '@yandex-cloud/uikit';
import {
   AutoGrowTextArea,
   BottomFooter,
   getSetDifference,
   isEmpty,
   isEqual,
   Loader,
   modalService,
   useDebouncedValue,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback, useLayoutEffect, useMemo, useRef, useState } from 'react';
import { useStore } from 'react-redux';
import { Link } from 'react-router-dom';
import { finalize, takeUntil } from 'rxjs/operators';

import { LegoButton, LegoCheckBox } from '../../../../_lego';
import { urlBuilder } from '../../../../models';
import { DeployUnit, PerLocationStrategy, Stage, StageConverter, StagePatcher } from '../../../../models/ui';

import { ConfirmationType, StageConfirmations } from '../../../../models/ui/Confirmations';
import { StagePatcherVisitor } from '../../../../models/ui/stage/StagePatcherVisitor';
import { SecretResolver, useActualSecretTokens } from '../../../../modules/secrets';
import { TStage } from '../../../../proto-typings';
import { pluralNumber, toYaml } from '../../../../utils';
import { DiffView, SimpleModal } from '../../../lib';
import { GetStageDeployParamsFlags } from '../../handlersGenerator';
import { useEndpointsInAwacsChecker } from '../../hooks';
import { StageHugeFormMode } from '../../models';

import classes from './StageDiffView.module.css';

interface Props {
   duConfirmations?: StageConfirmations;
   description?: string;
   isApprovalRequired: boolean;

   /**
    * Сырой стейдж последней (актуальной) версии (именно он показывается в левой части)
    */
   latestRawStage: TStage | undefined;

   mode: StageHugeFormMode;

   /**
    * Сырой стейдж, который будет пропатчен и сохранён
    */
   specValue: TStage;

   value: Stage;

   onCancel(): void;

   onDeploy(description: string, visitor: StagePatcherVisitor, flags?: GetStageDeployParamsFlags): void;

   onDeployDraft?(description: string, visitor: StagePatcherVisitor): void;
}

function getDuChanged(oldStage: Stage, newStage: Stage, predicate: (du: DeployUnit) => boolean): DeployUnit[] {
   const getDuList = (stage: Stage) => stage.deployUnits.filter(predicate);

   const oldDuList = getDuList(oldStage);
   const oldDuMap = new Map(oldDuList.map(du => [du.id, du]));
   const oldDuIdSet = new Set(oldDuMap.keys());

   const newDuList = getDuList(newStage);
   const newDuMap = new Map(newDuList.map(du => [du.id, du]));
   const newDuIdSet = new Set(newDuMap.keys());

   const { unchanged, added } = getSetDifference(oldDuIdSet, newDuIdSet);

   const list: DeployUnit[] = [];

   list.push(...[...added.values()].map(id => newDuMap.get(id)!));

   for (const id of unchanged.values()) {
      if (!isEqual(oldDuMap.get(id), newDuMap.get(id))) {
         list.push(newDuMap.get(id)!);
      }
   }

   return list;
}

export const StageDiffView: React.FC<Props> = React.memo(
   ({
      description: initialDescription = '',
      duConfirmations,
      isApprovalRequired,
      latestRawStage,
      mode,
      onCancel,
      onDeploy,
      onDeployDraft,
      specValue,
      value,
   }) => {
      const asIs = mode === StageHugeFormMode.ApplyAsIs;
      const store = useStore();
      const dismounted = useDismounted();

      const [originalDescription, setDescription] = useState(initialDescription);
      const description = useDebouncedValue(originalDescription, 250);

      const [asyncCheckingMessage, setAsyncCheckingMessage] = useState('');
      const endpointsInAwacsChecker = useEndpointsInAwacsChecker(false);

      const duWithSecretsMigration = useMemo(
         () =>
            new Set(
               Object.keys(duConfirmations ?? {}).filter(key =>
                  duConfirmations?.[key].has(ConfirmationType.SecretsMigration),
               ),
            ),
         [duConfirmations],
      );

      const [reloadTokens, setReloadTokens] = useState(false);
      const [tokenStore, isTokensLoading, secretCount] = useActualSecretTokens(
         !reloadTokens,
         value,
         duWithSecretsMigration,
      );

      const visitor = useMemo(
         () =>
            new StagePatcherVisitor({
               duConfirmations,
               secretResolver: new SecretResolver(store.getState(), tokenStore),
               initialStageId: value.initialId ?? value.id,
            }),
         [duConfirmations, store, tokenStore, value.id, value.initialId],
      );

      const getEditedSpec = useCallback(
         () =>
            StagePatcher.prepareToSave({
               asIs,
               description,
               editedValue: value,
               increment: true,
               latestRawStage,
               rawStage: specValue,
               visitor,
            }),
         [asIs, description, latestRawStage, specValue, value, visitor],
      );

      const wrapperRef = useRef<HTMLDivElement>(null); // Для расчета высоты футера
      const stageToYaml = useCallback((s: TStage) => toYaml(StagePatcher.removeUnsavedParts(s)), []);

      const beforeText = useMemo(
         () => (mode === StageHugeFormMode.New ? '' : stageToYaml(latestRawStage ?? specValue)),
         [stageToYaml, mode, latestRawStage, specValue],
      );
      const afterText = useMemo(() => stageToYaml(getEditedSpec()), [getEditedSpec, stageToYaml]);

      const oldValue = useMemo(() => StageConverter.fromApi(specValue), [specValue]);
      const duChangedWithPerLocation = useMemo(
         () => getDuChanged(oldValue, value, du => du.perLocationSettings.strategy === PerLocationStrategy.Sequential),
         [oldValue, value],
      );

      const allowParallelDeploying = duChangedWithPerLocation.length > 0;

      // region handlers
      const showCheckError = useCallback(
         (children: ReactNode, title: string, dataE2e?: string) => {
            modalService
               .open(SimpleModal, { children, title, dataE2e })
               .pipe(takeUntil(dismounted))
               .subscribe(
                  () => null,
                  () => null,
               );
         },
         [dismounted],
      );

      const doDeploy = useCallback(
         (draft = false, parallel = false) => {
            setAsyncCheckingMessage('Checking used AWACS balancers...');

            endpointsInAwacsChecker(value, specValue)
               .pipe(
                  finalize(() => setAsyncCheckingMessage('')),
                  takeUntil(dismounted),
               )
               .subscribe(checkResult => {
                  if (checkResult.isValid) {
                     if (draft) {
                        onDeployDraft?.(description, visitor);
                     } else {
                        onDeploy(description, visitor, parallel && { parallel: true });
                     }

                     return;
                  }

                  showCheckError(checkResult.error, 'Fix this before deleting location', 'remove-warning-modal');
               });
         },
         [
            endpointsInAwacsChecker,
            value,
            specValue,
            dismounted,
            showCheckError,
            onDeployDraft,
            description,
            visitor,
            onDeploy,
         ],
      );

      const handleDeploy = useCallback(() => doDeploy(false), [doDeploy]);
      const handleParallelDeploy = useCallback(() => doDeploy(false, true), [doDeploy]);

      const handleDeployDraft = useCallback(() => doDeploy(true), [doDeploy]);

      const handleChangeReloadTokens = useCallback(() => setReloadTokens(v => !v), []);
      // endregion

      useLayoutEffect(() => {
         // Перерасчет высоты футера при изменении description (могут быть добавлены строки в textarea)
         // Нужно для пересчета оставшейся высоты редактора
         if (!wrapperRef.current) {
            return;
         }

         const footer = wrapperRef.current.querySelector(`.${classes.bottomFooter}`);
         if (!footer) {
            return;
         }

         const property = '--bottom-footer-height';
         const previousValue = wrapperRef.current.style.getPropertyValue(property);
         const newValue = `${footer.getBoundingClientRect().height + 1}px`;
         if (previousValue === newValue) {
            return;
         }

         wrapperRef.current.style.setProperty(property, newValue);
      }, [description]);

      // region render
      const leftToolbar = secretCount > 0 && (
         <LegoCheckBox
            checked={reloadTokens}
            onChange={handleChangeReloadTokens}
            disabled={isTokensLoading}
            controlAttrs={{ 'data-e2e': 'StageDiffView:ReloadDelegationTokensCheckbox' }}
         >
            Reload {pluralNumber(secretCount, 'delegation token', 'delegation tokens')}
            <Loader inline={true} visible={isTokensLoading} />
         </LegoCheckBox>
      );

      return (
         <div className={classes.wrapper} ref={wrapperRef} data-e2e={'StageDiffView'}>
            <div className={classes.diffWrapper}>
               <DiffView
                  fullContext={mode === StageHugeFormMode.New}
                  language={'yaml'}
                  leftToolbar={leftToolbar}
                  modified={afterText}
                  original={beforeText}
                  unified={mode === StageHugeFormMode.New}
               />
            </div>

            <BottomFooter className={classes.bottomFooter}>
               <div data-e2e={'StageDiffView:Description'}>
                  <AutoGrowTextArea
                     value={originalDescription}
                     onUpdate={setDescription}
                     placeholder={'Release description'}
                     rows={2}
                  />
               </div>

               <div className={classes.buttons}>
                  {isApprovalRequired ? (
                     <LegoButton
                        controlAttrs={{ 'data-e2e': 'StageDiffView:SaveDraftButton' }}
                        disabled={originalDescription !== description}
                        onClick={handleDeployDraft}
                        theme={'action'}
                     >
                        Save as Draft
                     </LegoButton>
                  ) : (
                     <>
                        <LegoButton
                           controlAttrs={{ 'data-e2e': 'StageDiffView:DeployButton' }}
                           disabled={!isEmpty(asyncCheckingMessage) || originalDescription !== description}
                           onClick={handleDeploy}
                           theme={'action'}
                        >
                           Deploy
                        </LegoButton>
                        {asIs && allowParallelDeploying && (
                           <>
                              <LegoButton
                                 controlAttrs={{ 'data-e2e': 'StageDiffView:ParallelDeployButton' }}
                                 disabled={!isEmpty(asyncCheckingMessage) || originalDescription !== description}
                                 onClick={handleParallelDeploy}
                                 theme={'action'}
                              >
                                 Ignore per location and Deploy
                              </LegoButton>
                              <HelpPopover
                                 placement={'top'}
                                 content={
                                    <>
                                       Changed deploy units with per location settings:
                                       <ul>
                                          {duChangedWithPerLocation.map(du => (
                                             <li key={du.id}>{du.id}</li>
                                          ))}
                                       </ul>
                                    </>
                                 }
                              />
                           </>
                        )}
                     </>
                  )}

                  {mode !== StageHugeFormMode.ApplyAsIs && (
                     <LegoButton
                        controlAttrs={{ 'data-e2e': 'StageDiffView:CancelButton' }}
                        disabled={originalDescription !== description}
                        onClick={onCancel}
                     >
                        Edit
                     </LegoButton>
                  )}

                  <Loader
                     visible={!isEmpty(asyncCheckingMessage) || originalDescription !== description}
                     inline={true}
                     text={asyncCheckingMessage}
                     align={'left'}
                  />

                  {isApprovalRequired && (
                     <>
                        Approval policy is applied, so creation drafts is only available option. Drafts are available{' '}
                        <Link to={urlBuilder.stageDeployTickets(value.id)} target={'_blank'}>
                           here
                        </Link>
                        .
                     </>
                  )}
               </div>
            </BottomFooter>
         </div>
      );
      // endregion
   },
);

StageDiffView.displayName = 'StageDiffView';
