import React, { useCallback } from 'react';

import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';
import { DISMISS_REASON, IModalProps, ModalLayout } from '@yandex-infracloud-ui/libs';
import { FormikErrors } from 'formik';
import { useHistory } from 'react-router';

import { ypViewConfig, YpViewPart } from '../../../../models/ui';

import classes from './StageErrorsModal.module.css';

// TODO: подумать, может вынести в общее место
const id70 = '[\\w-]{1,70}';
const id64 = '[\\w-@:.]{1,64}';

const stageRegex = new RegExp(`stages\\/(${id70})`);
const duRegex = new RegExp(`${ypViewConfig.deployUnit.smallPrefix}-(${id70})`);
const boxRegex = new RegExp(`${ypViewConfig.box.smallPrefix}-(${id64})`);
const workloadRegex = new RegExp(`${ypViewConfig.workload.smallPrefix}-(${id64})`);

type ErrorItem = {
   path: string;
   messages: string[];
};

function getErrorList(errors: FormikErrors<unknown>, prefix = ''): ErrorItem[] {
   if (typeof errors === 'string') {
      return [{ path: prefix, messages: [errors] }];
   }
   if (Array.isArray(errors)) {
      return errors.flatMap((error, i) => getErrorList(error, `${prefix}[${i}]`));
   }
   if (typeof errors === 'object' && errors !== null) {
      return Object.keys(errors).flatMap(key =>
         getErrorList((errors as any)[key], `${prefix}${prefix ? '.' : ''}${key}`),
      );
   }
   return [];
}

function getYpViewData(formPath: string): { id: string; part: YpViewPart } {
   const [, stage] = formPath.match(stageRegex) ?? [];
   const [, deployUnit] = formPath.match(duRegex) ?? [];
   const [, box] = formPath.match(boxRegex) ?? [];
   const [, workload] = formPath.match(workloadRegex) ?? [];
   let part: YpViewPart = YpViewPart.stage;
   if (workload) {
      part = YpViewPart.workload;
   } else if (box) {
      part = YpViewPart.box;
   } else if (deployUnit) {
      part = YpViewPart.deployUnit;
   }
   return { part, id: workload ?? box ?? deployUnit ?? stage ?? 'stage' };
}

interface Props extends IModalProps<void> {
   errors: Record<string, FormikErrors<unknown>>;
}

export const StageErrorsModal: React.FC<Props> = React.memo(({ errors, ok, cancel }) => {
   const handleOk = useCallback(() => ok(), [ok]);
   const handleDismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);
   const history = useHistory();

   return (
      <ModalLayout
         title={'Please, correct all errors'}
         onDismiss={handleDismiss}
         onOk={handleOk}
         showFooter={true}
         okLabel={'Ignore, show diff'}
      >
         <div className={classes.errors}>
            {Object.keys(errors).map(formPath => {
               const error = errors[formPath];
               const { part, id } = getYpViewData(formPath);
               const config = ypViewConfig[part];
               return (
                  <div key={formPath} className={classes.nodeBlock}>
                     <h3>
                        <FontAwesomeIcon icon={config.fontAwesomeIcon} color={config.color} /> {id}
                     </h3>
                     {getErrorList(error).map(item => (
                        <div className={classes.fieldBlock} key={item.path}>
                           <div>{item.path}</div>
                           {item.messages.map(message => (
                              <div key={message} className={classes.message}>
                                 {message}
                              </div>
                           ))}
                        </div>
                     ))}

                     <Button
                        onClick={() => {
                           cancel(DISMISS_REASON);
                           // должно совпадать с существующими урлами!
                           // пока это так, так как id форм совпадает с урлами этих форм
                           history.push(formPath);
                        }}
                     >
                        Fix in form
                     </Button>
                  </div>
               );
            })}
         </div>
      </ModalLayout>
   );
});

StageErrorsModal.displayName = 'StageErrorsModal';
