import { deepClone, Hint, isEmpty, plural } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useEffect, useMemo, useState } from 'react';

import { LegoCheckBox } from '../../../../_lego';
import { DiskType, SidecarsForUpdating, sidecarsUpdateConfig, Stage } from '../../../../models/ui';
import { Confirmation, ConfirmationType, StageConfirmations } from '../../../../models/ui/Confirmations';
import { DeployUnitSecret } from '../../../../modules/secrets';

import {
   DiskIsolationHint,
   HddBandwidthHint,
   NetworkBandwidthHint,
   PatchersRevisionHint,
   SecretsMigrationHint,
   SidecarLabelRevisionHint,
   SsdBandwidthHint,
} from './hints';

import classes from './StagePatcherOptions.module.css';

export const getStageConfirmations = (stage: Stage, secrets: { [deployUnitId: string]: DeployUnitSecret[] }) => {
   const possibleConfirmations: Confirmation[] = [
      {
         id: ConfirmationType.SecretsMigration,
         title: 'Migrate old secrets',
         description: SecretsMigrationHint(),
      },
      {
         id: ConfirmationType.PatchersRevisionUpdate,
         title: 'PatchersRevision',
         description: PatchersRevisionHint(),
      },
      {
         id: ConfirmationType.HddBandwidth,
         title: 'Update HDD I/O',
         description: HddBandwidthHint(),
      },
      {
         id: ConfirmationType.SsdBandwidth,
         title: 'Update SSD I/O',
         description: SsdBandwidthHint(),
      },
      {
         id: ConfirmationType.DiskIsolation,
         title: 'Enable Disk Isolation',
         description: DiskIsolationHint(),
      },
      {
         id: ConfirmationType.NetworkBandwidthUpdate,
         title: 'Update Network Bandwidth Guarantee',
         description: NetworkBandwidthHint(),
      },
   ];

   for (const sidecarId of SidecarsForUpdating) {
      const sidecar = sidecarsUpdateConfig[sidecarId];
      possibleConfirmations.push({
         id: sidecar.confirmationType,
         title: `Update ${sidecar.title}`,
         description: SidecarLabelRevisionHint({ sidecarTitle: sidecar.title }),
      });
   }

   const confirmationList: Confirmation[] = [];

   const deployUnitsConfirmations = (stage.deployUnits ?? []).reduce((result, du) => {
      const confirmations = new Set<ConfirmationType>();

      const duId = du.initialId ?? du.id;
      const duSecrets = secrets?.[duId];

      if (duSecrets?.some(secret => secret.versions.some(v => v.legacy === true && !v.removed))) {
         confirmations.add(ConfirmationType.SecretsMigration);
      }

      // если есть лейбл, а значение в спеке не задано или оно меньше лейбла (аналогично #DEPLOY-5632)
      if (
         du.patchersRevision.label &&
         (!du.patchersRevision.value || du.patchersRevision.value < du.patchersRevision.label)
      ) {
         confirmations.add(ConfirmationType.PatchersRevisionUpdate);
      }

      if (du.disks.some(disk => disk.type === DiskType.HDD && !disk.bandwidth.guarantee)) {
         confirmations.add(ConfirmationType.HddBandwidth);
      }

      if (du.disks.some(disk => disk.type === DiskType.SSD && !disk.bandwidth.guarantee)) {
         confirmations.add(ConfirmationType.SsdBandwidth);
      }

      if (!du.tempDiskIsolation) {
         confirmations.add(ConfirmationType.DiskIsolation);
      }

      if (!du.networkBandwidth.guarantee && (!du.nodeSegmentId || du.nodeSegmentId === 'default')) {
         confirmations.add(ConfirmationType.NetworkBandwidthUpdate);
      }

      for (const sidecarId of SidecarsForUpdating) {
         const { resourceRevision, labelRevision } = du.sidecars[sidecarId];

         // если есть лейбл, а значение в спеке не задано или оно меньше лейбла #DEPLOY-5632
         if (labelRevision && (!resourceRevision || resourceRevision < labelRevision)) {
            confirmations.add(sidecarsUpdateConfig[sidecarId].confirmationType);
         }
      }
      if (confirmations.size > 0) {
         result[du.id] = confirmations;
      }
      return result;
   }, {} as StageConfirmations);

   possibleConfirmations.forEach(action => {
      if (Object.values(deployUnitsConfirmations).some(confirmations => confirmations.has(action.id))) {
         confirmationList.push(action);
      }
   });

   return {
      confirmationList,
      deployUnitsConfirmations,
      needUpdates: !isEmpty(confirmationList),
   };
};

interface Props {
   confirmationList: Confirmation[];
   deployUnitsConfirmations: StageConfirmations;
   needUpdates: boolean;
   onChange(v: StageConfirmations): void;
}

export const StagePatcherOptions: React.FC<Props> = React.memo(
   ({ confirmationList, deployUnitsConfirmations, needUpdates, onChange }) => {
      const [confirmed, setConfirmed] = useState(deepClone(deployUnitsConfirmations));

      useEffect(() => {
         setConfirmed(deepClone(deployUnitsConfirmations));
      }, [deployUnitsConfirmations]);

      useEffect(() => {
         onChange(confirmed);
      }, [confirmed, onChange]);

      const handleConfirmation = useCallback(
         (duId: string, actionId: ConfirmationType) => {
            if (!confirmed[duId]) {
               confirmed[duId] = new Set<ConfirmationType>();
            }

            const duActions = confirmed[duId];

            if (duActions.has(actionId)) {
               duActions.delete(actionId);
            } else {
               duActions.add(actionId);
            }

            setConfirmed({ ...confirmed });
         },
         [confirmed],
      );

      const handleGroupConfirmation = useCallback(
         (actionId: ConfirmationType) => {
            // если хотя бы в одном du включена галочка
            const someConfirmed = Object.values(confirmed).some(confirmations => confirmations.has(actionId));

            for (const duId of Object.keys(deployUnitsConfirmations)) {
               if (!deployUnitsConfirmations[duId].has(actionId)) {
                  // eslint-disable-next-line no-continue
                  continue;
               }

               if (!confirmed[duId]) {
                  confirmed[duId] = new Set<ConfirmationType>();
               }

               const duActions = confirmed[duId];

               if (someConfirmed) {
                  // сбрасываем галочки всем du
                  if (duActions.has(actionId)) {
                     duActions.delete(actionId);
                  }
               } else {
                  // выставляем галочки всем du
                  duActions.add(actionId);
               }
            }

            setConfirmed({ ...confirmed });
         },
         [confirmed, deployUnitsConfirmations],
      );

      const groupCheckboxProps = useCallback(
         (actionId: ConfirmationType) => {
            const confirmedActions = Object.values(confirmed).filter(confirmations => confirmations.has(actionId))
               .length;
            const totalActions = Object.values(deployUnitsConfirmations).filter(confirmations =>
               confirmations.has(actionId),
            ).length;
            const checked = confirmedActions === totalActions;

            const props = {
               checked,
               indeterminate: undefined as undefined | boolean,
            };

            if (confirmedActions === 0) {
               props.checked = false;
            } else if (!checked) {
               props.indeterminate = true;
            }

            return props;
         },
         [confirmed, deployUnitsConfirmations],
      );

      const duCount = useMemo(() => Object.keys(deployUnitsConfirmations).length, [deployUnitsConfirmations]);

      if (!needUpdates) {
         return (
            <div className={classes.wrapper} data-e2e={'StagePatcherOptions'}>
               Everything is up to date!
            </div>
         );
      }

      return (
         <div className={classes.wrapper} data-e2e={'StagePatcherOptions'}>
            <h2>We advise you to make following updates to your deploy {plural(duCount, 'unit', 'units')}.</h2>

            <p>
               These changes will update spec of your stage, you can verify it in the next step.
               <br />
               Beware that these updates will cause a <b>redeploy of all pods</b> in selected deploy units.
            </p>

            <table className={classes.table} data-e2e={'StagePatcherOptions:Table'}>
               <thead>
                  <tr>
                     <th>Deploy Unit</th>

                     {confirmationList.map(confirmation => (
                        <th key={`key-${confirmation.id}`} data-e2e={`Action:${confirmation.id}`}>
                           <LegoCheckBox
                              {...groupCheckboxProps(confirmation.id)}
                              onChange={() => handleGroupConfirmation(confirmation.id)}
                           >
                              {' '}
                              {confirmation.title} <Hint text={<>{confirmation.description}</>} />
                           </LegoCheckBox>
                        </th>
                     ))}
                  </tr>
               </thead>

               <tbody>
                  {Object.keys(deployUnitsConfirmations).map(duId => {
                     const duConfirmations = deployUnitsConfirmations[duId];

                     return (
                        <tr key={duId}>
                           <td data-e2e={'DeployUnitId'}>{duId}</td>
                           {/* <td>deployUnit012345678901234567890123456789012345678901234567890123456789</td> */}

                           {confirmationList.map(({ id }) => (
                              <td key={id} data-e2e={`Action:${duId}:${id}`}>
                                 {duConfirmations.has(id) ? (
                                    <LegoCheckBox
                                       // в confirmed может быть пусто
                                       checked={confirmed?.[duId]?.has(id)}
                                       onChange={() => handleConfirmation(duId, id)}
                                    />
                                 ) : null}
                              </td>
                           ))}
                        </tr>
                     );
                  })}
               </tbody>
            </table>
         </div>
      );
   },
);

StagePatcherOptions.displayName = 'StagePatcherOptions';
