import React, { Fragment, memo, ReactNode, useEffect, useMemo, useState } from 'react';

import { faChevronDown, faChevronUp, faLongArrowRight } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button, Card, Checkbox, HelpPopover, Spin } from '@yandex-cloud/uikit';
import { classNames, ExternalLink, getSetDifference } from '@yandex-infracloud-ui/libs';

import { EXTERNAL_LINKS } from '../../../../models';
import { QuotaRequestKeys, QuotaResourcesStore } from '../../../../models/ui/quota';
import { stageQuotaResourceGroups } from '../../../../models/ui/stage/resources';
import { ResourceGroup, ResourceGroupType, resourcesModel } from '../../../../modules/resources/config';
import { useAbcService, useNetworkErrors, useNetworkRequests } from '../../../../redux';
import { useConfig } from '../../../../services';
import { createKey, restoreObjectFromKey, Tree } from '../../../../utils';
import { OverQuota } from './OverQuota';
import { StageQuotaCell } from './StageQuotaCell';
import { NetworkItem, RequestState } from '../../../../redux/slices/network/model';
import { YpErrorTooltip } from '../../../network';

import classes from './StageQuota.module.css';

interface Props {
   tree: Tree<['resourceGroup', 'resource', 'location', 'metric'], number> | null;
   clusters: string[];
   editableResources: {
      old: QuotaResourcesStore;
      new: QuotaResourcesStore;
   };
   abc: string;
   initialAbc: string | null;
   viewMode: boolean;
}

export const StageQuota: React.FC<Props> = memo(
   ({ tree, clusters: clustersList, editableResources, abc, initialAbc, viewMode }) => {
      const [opened, setOpened] = useState(false);
      const [withDetails, setDetails] = useState(false);
      const [changeFlag, setChangeFlag] = useState(false);

      const initialAbcId = useMemo(() => initialAbc?.split(':')?.[2], [initialAbc]);
      const abcId = useMemo(() => abc?.split(':')?.[2], [abc]);

      const initialAbcService = useAbcService(initialAbcId ?? null);
      const abcService = useAbcService(abcId ?? null);

      const { clusters: defaultClusters } = useConfig()!;
      const clusters = useMemo(() => {
         const defaultClustersList = defaultClusters.map(e => e.value);
         const clustersSet = new Set(clustersList);
         const defaultClustersSet = new Set(defaultClustersList);
         return [
            ...defaultClustersList.filter(e => clustersSet.has(e)),
            ...clustersList.filter(e => !defaultClustersSet.has(e)).sort(),
         ];
      }, [clustersList, defaultClusters]);

      const unused = (resourceGroup: ResourceGroup, resourceName: string) =>
         clusters.every(cluster => {
            const key = createKey({ location: cluster, resourceGroup, resource: resourceName });
            return (editableResources.new[key] ?? 0) === 0 && (editableResources.old[key] ?? 0) === 0;
         });

      useEffect(() => {
         setChangeFlag(true);
         const timeout = setTimeout(() => setChangeFlag(false), 500);

         return () => {
            clearTimeout(timeout);
         };
      }, [editableResources.old, editableResources.new]);

      const requestKeys = useMemo(() => clustersList.map(cluster => QuotaRequestKeys.getOne({ cluster, abc })), [
         abc,
         clustersList,
      ]);
      const networkErrors = useNetworkErrors(requestKeys);
      const errorList = useMemo(() => Object.values(networkErrors).filter<Partial<NetworkItem>>(Boolean as any), [
         networkErrors,
      ]);

      const networkRequests = useNetworkRequests(requestKeys);
      const isLoading = useMemo(
         () => Object.values(networkRequests).some(request => request?.state === RequestState.PENDING || !request),
         [networkRequests],
      );

      const arrow = <FontAwesomeIcon icon={faLongArrowRight} className={classes.arrow} />;

      const { removed, added } = getSetDifference(
         new Set(Object.keys(editableResources.old)),
         new Set(Object.keys(editableResources.new)),
      );

      // есть хотя бы одно изменение
      const isChangeStageQuota =
         removed.size > 0 ||
         added.size > 0 ||
         Object.keys(editableResources.old).some(key => editableResources.old[key] !== editableResources.new[key]);

      // есть превышение квоты
      const existOverQuota =
         tree &&
         Object.keys(editableResources.new).some(key => {
            const { location, resourceGroup, resource } = restoreObjectFromKey(key);
            const oldValue = editableResources.old[key] ?? 0;
            const newValue = editableResources.new[key] ?? 0;
            const { usage = 0, total = 0 } = tree[resourceGroup]?.[resource]?.[location] ?? {};

            const actualUsage = usage - oldValue;
            const available = total - actualUsage;

            return newValue > available;
         });

      const tdWrap = (
         <>
            {withDetails && <div />}
            <div />
         </>
      );

      let bodyContent: ReactNode = null;
      if (isLoading) {
         bodyContent = <Spin size={'l'} />;
      } else if (tree) {
         bodyContent = (
            <table className={classes.table}>
               <thead>
                  <tr>
                     <th>Resource</th>
                     {clusters.map(cluster => (
                        <th key={cluster} data-test={'StageQuota:LocationTitle'}>
                           {cluster.toUpperCase()}
                        </th>
                     ))}
                  </tr>
               </thead>
               <tbody>
                  {stageQuotaResourceGroups.map(resourceGroup => {
                     if (!resourcesModel.hasOwnProperty(resourceGroup)) {
                        return null;
                     }
                     const { resource, groupTitle, groupType } = resourcesModel[resourceGroup]!;
                     const multi = groupType === ResourceGroupType.Multi;
                     const resourceGroupData = tree[resourceGroup];
                     if (!resourceGroupData) {
                        return null;
                     }

                     const empty = Object.keys(resourceGroupData).every(name => unused(resourceGroup, name));
                     if (empty) {
                        return null;
                     }

                     const resourceNames = Object.keys(resourceGroupData).sort();

                     const { getTitle } = resource;

                     return (
                        <tr key={resourceGroup}>
                           <td>
                              <div className={classes.resourceGroup}>
                                 <div>
                                    <div>{groupTitle}</div>
                                    {tdWrap}
                                 </div>
                                 {multi && (
                                    <div className={classes.resourceNames}>
                                       {resourceNames.map(resourceName =>
                                          unused(resourceGroup, resourceName) ? null : (
                                             <Fragment key={resourceName}>
                                                <div>{getTitle(resourceName)}</div>
                                                {tdWrap}
                                             </Fragment>
                                          ),
                                       )}
                                    </div>
                                 )}
                              </div>
                           </td>
                           {clusters.map(cluster => (
                              <td key={cluster}>
                                 {resourceNames.map(resourceName => {
                                    if (unused(resourceGroup, resourceName)) {
                                       return null;
                                    }
                                    const resourceData = resourceGroupData[resourceName];
                                    const { total = 0, usage = 0 } = resourceData?.[cluster] ?? {};
                                    const key = createKey({ location: cluster, resourceGroup, resource: resourceName });
                                    const oldValue = editableResources.old[key] ?? 0;
                                    const newValue = editableResources.new[key] ?? 0;

                                    return (
                                       <Fragment key={resourceName}>
                                          <StageQuotaCell
                                             {...{
                                                usage,
                                                total,
                                                newValue,
                                                oldValue,
                                                resourceGroup,
                                                viewMode,
                                                withDetails,
                                             }}
                                          />
                                       </Fragment>
                                    );
                                 })}
                              </td>
                           ))}
                        </tr>
                     );
                  })}
               </tbody>
            </table>
         );
      } else {
         bodyContent = (
            <div>
               <p>No quota info</p>
            </div>
         );
      }

      return (
         <div className={classes.plate}>
            <Card view={'raised'} className={classNames(classes.card, { [classes.change]: changeFlag })}>
               <div className={classes.header}>
                  <h2>
                     <span>Resource quota{isChangeStageQuota && <sup title={'changed'}>*</sup>}</span>
                     <HelpPopover
                        className={classes.help}
                        placement={opened ? ['bottom', 'left'] : 'top'}
                        content={
                           <>
                              <ExternalLink href={EXTERNAL_LINKS.deployDocs.quota}>documentation</ExternalLink>
                              {opened && (
                                 <div className={classes.format}>
                                    <StageQuotaCell
                                       usage={124000}
                                       total={161630}
                                       oldValue={1200}
                                       newValue={viewMode ? 1200 : 2300}
                                       resourceGroup={ResourceGroup.Cpu}
                                       viewMode={false}
                                       withDetails={true}
                                    />
                                    <div>
                                       current {!viewMode && <span>{arrow} new value</span>}
                                       <br />
                                       available quota
                                       <br />
                                       usage / total
                                    </div>
                                 </div>
                              )}
                           </>
                        }
                     />
                     <span className={classes.abc} data-test={'StageQuota:ABC'}>
                        {initialAbc && initialAbc !== abc && (
                           <span>
                              {initialAbcService ? (
                                 <ExternalLink
                                    href={EXTERNAL_LINKS.abcQuotaRequest(initialAbcService.id)}
                                    className={classes.abcLink}
                                 >
                                    {initialAbcService.name}
                                 </ExternalLink>
                              ) : (
                                 initialAbc
                              )}{' '}
                              {arrow}{' '}
                           </span>
                        )}
                        {abcService ? (
                           <ExternalLink href={EXTERNAL_LINKS.abcQuotaRequest(abcService.id)}>
                              {abcService.name}
                           </ExternalLink>
                        ) : (
                           abc
                        )}
                     </span>
                  </h2>
                  <div>
                     {existOverQuota && (
                        <div className={classes.overQuota} data-test={'StageQuota:OverQuota'}>
                           <OverQuota />
                           <span>Over quota</span>
                        </div>
                     )}
                  </div>
                  <div>
                     {opened && tree && (
                        <Checkbox onChange={() => setDetails(e => !e)} checked={withDetails}>
                           Details
                        </Checkbox>
                     )}
                  </div>
                  <Button
                     onClick={() => setOpened(e => !e)}
                     view={'flat'}
                     extraProps={{ 'data-test': 'StageQuota:ToggleButton' }}
                  >
                     <FontAwesomeIcon icon={opened ? faChevronDown : faChevronUp} />
                  </Button>
               </div>
               {opened && (
                  <>
                     {bodyContent}
                     {!isLoading &&
                        errorList.map(
                           ({ error, request }) => error && <YpErrorTooltip error={error} request={request} />,
                        )}
                  </>
               )}
            </Card>
         </div>
      );
   },
);

StageQuota.displayName = 'StageQuota';
