import { isEqual, modalService } from '@yandex-infracloud-ui/libs';
import { History } from 'history';
import { Dispatch } from 'redux';
import { Observable, of, Subject } from 'rxjs';
import { map, switchMap, takeUntil, tap } from 'rxjs/operators';
import { v4 as uuid } from 'uuid';

import { handleApiError, urlBuilder } from '../../models';
import { emptyRawStage, PerLocationStrategy, Stage, StagePatcher } from '../../models/ui';
import { StagePatcherVisitor } from '../../models/ui/stage/StagePatcherVisitor';
import Notifications from '../../old-code/services/Notifications';
import { TDeployUnitSpec_TDeploySettings_EDeployStrategy, TStage } from '../../proto-typings';
import { formDataSlice, getStages } from '../../redux';
import { ypApi } from '../../services';
import { StageDeploySettingsOptions } from '../../services/api/services/YpApi';

import { formStatesSlice, nodeStateSlice } from '../forms';
import { ReloadStageModal } from './components/ReloadStageModal/ReloadStageModal';
import { StageHugeFormMode } from './models';

// TODO нужно переделать (весь файл), слишком мудрено. Генератор генераторов, какой-то.
//  Набор из генераторов функций, которые на вход принимают функцию, которую кто-то внешний сгенерировал, используя один из генераторов (stageReset).
//  Хотя можно было напрямую вызвать. (rootFormId и stageId можно передать сразу)
//  Можно проще, да и теперь место одно (StageHugeForm), где это используется, поэтому шарить код такими костылями стало необоснованно.

interface GetStageResetParams {
   rootFormId: string;
}

export type GetStageDeployParamsFlag = 'parallel';
export type GetStageDeployParamsFlags = {
   [flag in GetStageDeployParamsFlag]?: boolean;
};

interface GetStageDeployParams {
   stageId: string;
   rawStage: TStage;
   latestRawStage: TStage | undefined;
   editedValue: Stage;
   stageReset: () => void;
   // revisionTimestamp?: number; TODO https://st.yandex-team.ru/DEPLOY-3723
}

interface GetStageDeployDraftParams {
   stageId: string;
   rawStage: TStage;
   editedValue: Stage;
   stageReset: () => void;
}

interface GetNewStageDeployDraftParams {
   editedValue: Stage;
   stageReset: () => void;
}

interface StageFormActionsConstructorParams {
   dismounted: Subject<unknown>;
   dispatch: Dispatch<any>;
   history: History<unknown>;
   mode: StageHugeFormMode;
   notifications: Notifications | undefined;
}

function perLocationToParallel(stageId: string, duIds: string[]) {
   ypApi
      .getStage(stageId, stage => [stage.spec.deploy_units])
      .subscribe(stage => {
         const deployUnits = stage?.spec?.deploy_units;
         if (!deployUnits) {
            return;
         }
         ypApi
            .overrideStageDeploySettings({
               stageId,
               duSettings: duIds
                  .filter(duId => deployUnits.hasOwnProperty(duId) && deployUnits[duId]?.revision)
                  .reduce((settings, duId) => {
                     settings[duId] = {
                        revision: deployUnits[duId]!.revision!,
                        settings: {
                           cluster_sequence: [],
                           deploy_strategy: TDeployUnitSpec_TDeploySettings_EDeployStrategy.PARALLEL,
                        },
                     };
                     return settings;
                  }, {} as StageDeploySettingsOptions['duSettings']),
            })
            .subscribe();
      });
}

export function handlersGenerator({
   dismounted,
   dispatch,
   history,
   mode,
   notifications,
}: StageFormActionsConstructorParams) {
   const asIs = mode === StageHugeFormMode.ApplyAsIs;

   function getStageResetHandler({ rootFormId }: GetStageResetParams) {
      return () => {
         // Очистка добавленных узлов (вручную или клонированием)
         dispatch(nodeStateSlice.actions.clearByPrefix(rootFormId));

         // Очистка форм в redux
         dispatch(formStatesSlice.actions.clearByPrefix(rootFormId));
      };
   }

   function reloadStageIfNeeded(stageId: string, rawStage: TStage): Observable<boolean> {
      return ypApi.isStageChanged(stageId, rawStage.spec?.revision).pipe(
         switchMap(changed => (changed ? modalService.open(ReloadStageModal, {}) : of(false))),
         switchMap(reload => (reload && rawStage.meta?.id ? ypApi.getStage(rawStage.meta.id) : of(null))),
         tap(newRawStage => {
            if (newRawStage) {
               dispatch(formDataSlice.actions.updateStageFormData({ formId: mode, rawStage: newRawStage as TStage }));
            }
         }),
         map(x => Boolean(x)),
      );
   }

   function getStageDeployHandler({
      editedValue,
      latestRawStage,
      rawStage,
      stageId,
      stageReset,
   }: GetStageDeployParams) {
      return (description: string, visitor: StagePatcherVisitor, flags: GetStageDeployParamsFlags = {}) => {
         reloadStageIfNeeded(stageId, rawStage!)
            .pipe(takeUntil(dismounted))
            .subscribe(reloaded => {
               if (reloaded === true) {
                  return;
               }

               const newStage = StagePatcher.prepareToSave({
                  asIs,
                  description,
                  editedValue,
                  increment: true,
                  latestRawStage,
                  rawStage,
                  visitor,
               });

               ypApi
                  .updateStage(
                     newStage,
                     rawStage?.meta?.project_id !== newStage.meta?.project_id,
                     rawStage?.meta?.account_id !== newStage.meta?.account_id,
                     !isEqual(rawStage?.labels?.tags, newStage.labels?.tags),
                     rawStage?.labels?.infra_service !== newStage.labels?.infra_service ||
                        rawStage?.labels?.infra_service_name !== newStage.labels?.infra_service_name ||
                        rawStage?.labels?.infra_environment !== newStage.labels?.infra_environment ||
                        rawStage?.labels?.infra_environment_name !== newStage.labels?.infra_environment_name,
                     // https://st.yandex-team.ru/DEPLOY-5195#62068a4d19c28f389b25b928
                     rawStage?.labels?.du_patchers_autoupdate_revision !==
                        newStage.labels?.du_patchers_autoupdate_revision,
                  )
                  .pipe(takeUntil(dismounted))
                  .subscribe(() => {
                     stageReset();
                     dispatch(getStages({ objectIds: [stageId] }));
                     if (flags.parallel) {
                        perLocationToParallel(
                           stageId,
                           editedValue.deployUnits
                              .filter(du => du.perLocationSettings.strategy === PerLocationStrategy.Sequential)
                              .map(du => du.id),
                        );
                     }
                     history!.push(urlBuilder.stageStatus(stageId));
                  }, handleApiError('Saving stage', notifications));
            });
      };
   }

   function getStageDeployDraftHandler({ stageReset, rawStage, editedValue, stageId }: GetStageDeployDraftParams) {
      return (description: string, visitor: StagePatcherVisitor) => {
         const newStage = StagePatcher.prepareToSave({
            asIs,
            description,
            editedValue,
            increment: false,
            rawStage,
            visitor,
         });
         const id = uuid();

         ypApi
            .createDraftAndTicket(id, newStage.meta!.id, newStage.spec!)
            .pipe(takeUntil(dismounted))
            .subscribe(() => {
               stageReset();
               history!.push(urlBuilder.deployTicket(stageId, `${id}-ticket`));
            }, handleApiError('Creating draft', notifications));
      };
   }

   function getNewStageDeployHandler({ stageReset, editedValue }: GetNewStageDeployDraftParams) {
      return (description: string, visitor: StagePatcherVisitor) => {
         const newStage = StagePatcher.prepareToSave({
            asIs,
            description,
            editedValue,
            increment: true,
            rawStage: emptyRawStage as TStage,
            visitor,
         });
         const stageId = newStage.meta!.id;

         ypApi
            .createStage(newStage)
            .pipe(takeUntil(dismounted))
            .subscribe(() => {
               stageReset();
               history!.push(urlBuilder.stageStatus(stageId));
            }, handleApiError('Creating stage', notifications));
      };
   }

   return {
      getStageResetHandler,
      getStageDeployHandler,
      getStageDeployDraftHandler,
      getNewStageDeployHandler,
   };
}
