import React, { ReactNode, useCallback } from 'react';
import { ExternalLink, getSetDifference } from '@yandex-infracloud-ui/libs';
import { forkJoin, Observable, of } from 'rxjs';
import { map, mergeMap } from 'rxjs/operators';

import { notEmpty } from '../../../utils';
import { EXTERNAL_LINKS } from '../../../models';
import { EndpointSetRecord, ListBalancersPayload } from '../../../models/api';
import { Stage, StageConverter } from '../../../models/ui';
import { TStage } from '../../../proto-typings';
import { awacsApi } from '../../../services';

function findUsedEndpointSet(stage: Stage) {
   const endpointSet: Record<string, EndpointSetRecord> = {};
   for (const du of stage.deployUnits) {
      for (const cluster of Object.keys(du.locations)) {
         const location = du.locations[cluster];

         if (location.enabled) {
            endpointSet[`${du.id}.${cluster}`] = { id: `${stage.id}.${du.id}`, cluster };
         }
      }
   }

   return endpointSet;
}

function allBalancersInNamespaceIsTesting(namespaceBalancerListPayload: ListBalancersPayload) {
   // FIXME: any уйдет как только я затащу актуальные типы для awacs #DEPLOY-5957
   return namespaceBalancerListPayload.balancers.every((balancer: any) => {
      if (!balancer.spec.envType.toLowerCase().includes('testing')) {
         return false;
      }

      const [, minor, patch] = balancer.spec.yandexBalancer.config.l7Macro.version.split('.').map(Number);
      if (minor < 3 || (minor === 3 && patch < 12)) {
         return false;
      }
      return true;
   });
}

interface CheckResult {
   isValid: boolean;
   error: ReactNode;
}

export function useEndpointsInAwacsChecker(
   isStageRemoving: boolean,
): (stage: Stage | null, rawStage: TStage) => Observable<CheckResult> {
   return useCallback(
      (stage: Stage | null, rawStage: TStage) => {
         if (isStageRemoving && stage) {
            throw new Error('Expected null if isStageRemoving');
         }

         if (!isStageRemoving && !stage) {
            throw new Error('Expected actual stage if isStageRemoving=false');
         }

         const previousValue = StageConverter.fromApi(rawStage);
         const previousEndpointSet = findUsedEndpointSet(previousValue);
         const actualEndpointSet = stage ? findUsedEndpointSet(stage) : {};
         const { removed } = getSetDifference(
            new Set(Object.keys(previousEndpointSet)),
            new Set(Object.keys(actualEndpointSet)),
         );
         const removedEndpointSets = Array.from(removed).map(k => previousEndpointSet[k]);

         return awacsApi.getNamespacesForEndpointSet(removedEndpointSets).pipe(
            mergeMap(namespaces => {
               const productionNamespaces = namespaces.map(namespace =>
                  awacsApi
                     .ListBalancersByNamespace(namespace)
                     .pipe(
                        map(balancerListPayload =>
                           allBalancersInNamespaceIsTesting(balancerListPayload) ? null : namespace,
                        ),
                     ),
               );

               return productionNamespaces.length
                  ? forkJoin(productionNamespaces).pipe(map(nmspaces => nmspaces.filter(notEmpty)))
                  : of([]);
            }),
            map(namespaces => {
               const isValid = namespaces.length === 0;

               return {
                  isValid,
                  error: isValid ? null : (
                     <>
                        Namespaces that are listed below, flow traffic to endpoint sets{' '}
                        {isStageRemoving ? 'of this stage' : 'you are going to delete'}:
                        <ul>
                           {namespaces.map(namespace => (
                              <li key={namespace}>
                                 <ExternalLink href={EXTERNAL_LINKS.awacsNamespace(namespace)}>
                                    {namespace}
                                 </ExternalLink>
                              </li>
                           ))}
                        </ul>
                     </>
                  ),
               } as CheckResult;
            }),
         );
      },
      [isStageRemoving],
   );
}
