import { isEmpty, modalService, useDismounted } from '@yandex-infracloud-ui/libs';
import { useCallback, useEffect, useRef, useState } from 'react';
import { useDispatch, useSelector, useStore } from 'react-redux';
import { useHistory } from 'react-router';
import { concat, Observable, of, timer } from 'rxjs';
import { filter, finalize, map, switchMap, takeUntil, tap } from 'rxjs/operators';

import { Stage } from '../../../models/ui';
import { secretsSlice } from '../../../modules/secrets';
import { RootStateWithSecrets, selectStageSecrets } from '../../../modules/secrets/slice';
import { TStage } from '../../../proto-typings';
import { formDataSlice, RootState } from '../../../redux';
import { ypApi } from '../../../services';
import { selectFormsHasUnsavedChanges, selectNodeList, selectNodesHasUnsavedChanges } from '../../forms';
import { getFormsValidationErrors, HugeFormRef } from '../../huge-form';

import { ReloadStageModal } from '../components/ReloadStageModal/ReloadStageModal';
import { StageErrorsModal } from '../components/StageErrorsModal/StageErrorsModal';
import { getStageConfirmations } from '../components/StagePatcherOptions/StagePatcherOptions';
import { StageHugeFormMode, StageHugeFormShowPage } from '../models';
import { AnySubForm } from '../../huge-form/models';

interface UseStageFormArgs {
   mode: StageHugeFormMode;
   rawStage: TStage;
   rootFormId: string;
   value: Stage;
}

/**
 * Хук для общих действий формы стейджа без привязки к конкретным функциям
 */
export function useStageForm({ rootFormId, mode, value, rawStage }: UseStageFormArgs) {
   // общие хуки
   const history = useHistory();
   const dismounted = useDismounted();
   const reduxStore = useStore<RootState>();
   const dispatch = useDispatch();

   // !Form или isSubmitting = форма сохранена, изменения не вносятся
   const [showPage, setShowPage] = useState<StageHugeFormShowPage>(
      mode === StageHugeFormMode.ApplyAsIs
         ? StageHugeFormShowPage.Diff
         : mode === StageHugeFormMode.Update
         ? StageHugeFormShowPage.Update
         : StageHugeFormShowPage.Edit,
   );
   const [isSubmitting, setIsSubmitting] = useState(false);

   // значение формы
   const [editedValue, setEditedValue] = useState<Stage>(value);

   // ссылка на форму, необходимо указать как значение ref
   const hugeFormRef = useRef<HugeFormRef<Stage>>(null);

   const secretsSelector = useCallback((s: RootStateWithSecrets) => selectStageSecrets(s, value.id), [value.id]);
   const secrets = useSelector(secretsSelector);

   const [stageConfirmations, setStageConfirmations] = useState(getStageConfirmations(value, secrets));

   const hasUnsavedChangesSelector = useCallback(
      (s: RootState) => selectFormsHasUnsavedChanges(s, rootFormId) || selectNodesHasUnsavedChanges(s, rootFormId),
      [rootFormId],
   );
   const hasUnsavedChanges = useSelector(hasUnsavedChangesSelector);

   const nodeMapSelector = useCallback((s: RootState) => selectNodeList(s, rootFormId), [rootFormId]);
   const nodeMap = useSelector(nodeMapSelector);

   // модальные окна, открываются перед показом страницы update
   // если скипаем update, то перед показом diff
   const reloadStageModal = useCallback(
      (stage: Stage): Observable<boolean> => {
         if (mode === StageHugeFormMode.New || mode === StageHugeFormMode.ApplyAsIs) {
            return of(false);
         }

         return ypApi.isStageChanged(stage.id, stage.revision).pipe(
            switchMap(changed => (changed ? modalService.open(ReloadStageModal, {}) : of(false))),
            switchMap(reload => (reload ? ypApi.getStage(stage.id) : of(null))),
            filter((result): result is TStage => result !== null),
            tap(newRawStage => {
               dispatch(formDataSlice.actions.updateStageFormData({ formId: mode, rawStage: newRawStage }));
            }),
            map(x => Boolean(x)),
         );
      },
      [dispatch, mode],
   );

   const errorWarningModal = useCallback(
      (customForms?: AnySubForm[], isVirtualForms?: boolean) => {
         const forms = customForms ?? hugeFormRef.current?.getForms();
         if (!forms || mode === StageHugeFormMode.ApplyAsIs || mode === StageHugeFormMode.Update) {
            return of(null);
         }

         let existedForms = forms;
         if (!isVirtualForms) {
            existedForms = forms.filter(f => {
               const node = nodeMap.get(f.id);

               return node ? node.mode !== 'removed' : true;
            });
         }

         return getFormsValidationErrors(rootFormId, existedForms, isVirtualForms ? null : reduxStore).pipe(
            switchMap(errors => (isEmpty(errors) ? of(null) : modalService.open(StageErrorsModal, { errors }))),
         );
      },
      [reduxStore, rootFormId, nodeMap, mode],
   );

   // сохраняет текущее состояние формы локально
   const saveFormValue = useCallback((stage: Stage) => {
      setEditedValue(stage);

      return stage;
   }, []);

   // сохраняет текущее значение и переходит в режим сохранения
   const openDiff = useCallback(
      (stage: Stage) => {
         if (saveFormValue(stage)) {
            setShowPage(StageHugeFormShowPage.Diff);
         }
      },
      [saveFormValue],
   );
   // сохраняет текущее значение и переходит в режим сохранения
   const openUpdates = useCallback(
      (st: Stage) => {
         const stage = saveFormValue(st);

         if (stage) {
            const confirmations = getStageConfirmations(stage, secrets);

            // обновляем, если при редактировании формы поменялись данные
            setStageConfirmations(confirmations);

            if (mode === StageHugeFormMode.ApplyAsIs || !confirmations.needUpdates) {
               openDiff(stage);
            } else {
               setShowPage(StageHugeFormShowPage.Update);
            }
         }
      },
      [mode, secrets, openDiff, saveFormValue],
   );

   const handleCancelUpdate = useCallback(() => {
      // только для двух страниц формы: Update/Diff
      if (showPage === StageHugeFormShowPage.Diff && mode === StageHugeFormMode.Update) {
         setShowPage(StageHugeFormShowPage.Update);
      } else {
         setShowPage(StageHugeFormShowPage.Edit);
      }
   }, [mode, showPage]);

   const handleUpdate = useCallback(
      (preparedStage?: Stage, forms?: AnySubForm[], isVirtualForms?: boolean) => {
         if (!hugeFormRef.current) {
            return;
         }
         const stage = preparedStage ?? hugeFormRef.current?.getValue();
         if (showPage === StageHugeFormShowPage.Update) {
            openDiff(stage);
         } else {
            setIsSubmitting(true);

            // timer() нужен, чтобы дать возможность React применить изменения, связанные с isSubmitting
            // перед потенциально тяжелой валидацией, блокирующей event loop (валидация больших стейджей)
            concat(timer(0), reloadStageModal(stage), errorWarningModal(forms, isVirtualForms))
               .pipe(
                  finalize(() => setIsSubmitting(false)),
                  takeUntil(dismounted),
               )
               .subscribe({
                  complete: () => {
                     openUpdates(stage);
                  },
                  error: handleCancelUpdate,
               });
         }
      },
      [reloadStageModal, errorWarningModal, dismounted, showPage, openDiff, openUpdates, handleCancelUpdate],
   );

   const loadStageSecrets = useCallback(() => {
      if (rawStage.spec) {
         dispatch(
            secretsSlice.actions.loadFromStage({
               stage: value,
               rawStageSpec: rawStage.spec,
               recreate: true,
            }),
         );
      }
   }, [dispatch, rawStage.spec, value]);

   const reloadSecretUsages = useCallback(() => {
      // Таймер здесь, очевидно, костыль. Он нужен, чтобы подождать,
      // когда форма актуализируется, чтобы из неё можно было достать свежее значение.
      // Теоретически это место может привести к "состоянию гонки",
      // но на практике значение достаточно большое, чтобы его избежать.
      timer(250)
         .pipe(takeUntil(dismounted))
         .subscribe(() => {
            if (hugeFormRef.current && rawStage.spec) {
               const stage = hugeFormRef.current.getValue(); // таймер для этого
               dispatch(
                  secretsSlice.actions.loadFromStage({
                     stage,
                     rawStageSpec: rawStage.spec,
                     recreate: false,
                  }),
               );
            }
         });
   }, [dismounted, dispatch, rawStage.spec]);

   const cloneDuInSecrets = useCallback(
      (levelId: string, originalInitialId: string, newId: string) => {
         if (hugeFormRef.current && levelId === 'deployUnit') {
            const stage = hugeFormRef.current.getValue();
            dispatch(
               secretsSlice.actions.cloneDeployUnit({
                  stageId: stage.initialId ?? stage.id,
                  originalInitialId,
                  newDuId: newId,
               }),
            );
         }
      },
      [dispatch],
   );

   const handleRevertChanges = useCallback(() => {
      if (hugeFormRef.current) {
         hugeFormRef.current.resetCurrentForm(); // reset formik form, rebuild all SubForm
      }

      history.push(rootFormId);

      loadStageSecrets();
   }, [history, loadStageSecrets, rootFormId]);

   useEffect(() => {
      setStageConfirmations(getStageConfirmations(value, secrets));
   }, [secrets, value]);

   useEffect(() => {
      if (mode === StageHugeFormMode.ApplyAsIs) {
         handleUpdate();
      }
   }, [handleUpdate, mode]);

   useEffect(() => {
      // Загрузка используемых в стейдже секретов
      loadStageSecrets();
   }, [loadStageSecrets]);

   return {
      cloneDuInSecrets,
      stageConfirmations,
      editedValue,
      handleCancelUpdate,
      handleRevertChanges,
      handleUpdate,
      hasUnsavedChanges,
      hugeFormRef,
      isSubmitting,
      reloadSecretUsages,
      showPage,
   };
}
/* function patcherOptionsModal(stage: Stage): any {
   throw new Error('Function not implemented.');
} */
