import { useMemo, useState } from 'react';
import {
   BoxFormParams,
   boxPaths,
   DeployUnitDisk,
   DeployUnitFormParams,
   DiskType,
   duPaths,
   Stage,
   stageLevels,
} from '../../../models/ui';
import { getStageDiskType } from '../../../models/ui/stage/resources';
import { createKey, formName, getEmptyStoreAction, isEmptyStoreAction, StoreAction, updateStore } from '../../../utils';
import { FormChangeListener, FormChangeType } from '../../huge-form';

interface GetStageDiskTypeStoreActionParams {
   levelId: string;
   duId: string;
   boxId: string;
   path: string;
   oldValue: unknown;
   newValue: unknown;
   disks: DeployUnitDisk[];
   type: FormChangeType;
}

function getStageDiskTypeAction({
   levelId,
   duId,
   boxId,
   path,
   newValue,
   disks,
   type,
}: GetStageDiskTypeStoreActionParams): StoreAction<DiskType> {
   const action = getEmptyStoreAction<DiskType>();
   const key = createKey({ deployUnit: duId, box: boxId });

   switch (levelId) {
      case stageLevels.deployUnit.id: {
         switch (path) {
            case formName(duPaths.disks): {
               const newDisks = newValue as DeployUnitFormParams['disks'];
               const duDiskType = newDisks[0]?.type;

               if (type !== FormChangeType.Remove && duDiskType) {
                  action.write[key] = { value: duDiskType };
               } else {
                  action.delete.push(key);
               }
               break;
            }
         }
         break;
      }
      case stageLevels.box.id: {
         switch (path) {
            // eslint-disable-next-line no-underscore-dangle
            case formName(boxPaths.virtualDiskIdRef): {
               const virtualDiskIdRef = newValue as BoxFormParams['virtualDiskIdRef'];
               const boxDisk = disks.find(disk => disk.id === virtualDiskIdRef);

               if (type !== FormChangeType.Remove && boxDisk) {
                  action.write[key] = { value: boxDisk.type };
               } else {
                  action.delete.push(key);
               }
               break;
            }
         }
      }
   }

   return action;
}

export function useStageFormDiskType(stage: Stage, isNewQuota: boolean) {
   const initStageDiskTypeStore = useMemo(() => getStageDiskType(stage), [stage]);

   const oldStageDiskTypeStore = useMemo(() => (isNewQuota ? {} : initStageDiskTypeStore), [
      initStageDiskTypeStore,
      isNewQuota,
   ]);

   const [newStageDiskTypeStore, setNewStageDiskTypeStore] = useState(initStageDiskTypeStore);

   const diskTypeChangeListener: FormChangeListener = useMemo(
      () => ({
         listenFields: {
            deployUnit: [formName(duPaths.disks)],
            // eslint-disable-next-line no-underscore-dangle
            box: [formName(boxPaths.virtualDiskIdRef)],
         },
         onChange({ levelId, path, newValue, oldValue, formValues, parentForms, type }) {
            const { id } = formValues;
            let duId = '';
            let boxId = '';
            let disks: DeployUnitDisk[] = [];

            switch (levelId) {
               case stageLevels.deployUnit.id: {
                  duId = id;
                  disks = formValues.disks;
                  break;
               }
               case stageLevels.box.id: {
                  boxId = id;
                  duId = parentForms[0].formParams.id;
                  disks = parentForms[0].formParams.disks;
                  break;
               }
            }

            const storeAction = getStageDiskTypeAction({
               levelId,
               duId,
               boxId,
               path,
               oldValue,
               newValue,
               disks,
               type,
            });

            if (!isEmptyStoreAction(storeAction)) {
               setNewStageDiskTypeStore(store => updateStore(store, storeAction));
            }
         },
      }),
      [],
   );

   return { newStageDiskTypeStore, oldStageDiskTypeStore, diskTypeChangeListener };
}
