import { getSetDifference } from '@yandex-infracloud-ui/libs';
import { useEffect, useMemo, useState } from 'react';
import { DeployUnitFormParams, Stage } from '../../../models/ui';
import { getStagePodCountStore } from '../../../models/ui/stage/resources';
import { createKey, getEmptyStoreAction, isEmptyStoreAction, StoreAction, updateStore } from '../../../utils';
import { FormChangeListener, FormChangeType } from '../../huge-form';

function getStagePodCountStoreAction(
   duId: string,
   oldLocations: DeployUnitFormParams['locations'],
   newLocations: DeployUnitFormParams['locations'],
   type: FormChangeType,
): StoreAction<number> {
   const action = getEmptyStoreAction<number>();
   const oldLocationSet = new Set(Object.keys(oldLocations).filter(e => oldLocations[e].enabled));
   const newLocationSet = new Set(Object.keys(newLocations).filter(e => newLocations[e].enabled));

   const { removed } = getSetDifference(oldLocationSet, newLocationSet);
   for (const locationName of removed.values()) {
      const locationKey = createKey({ deployUnit: duId, location: locationName });
      action.delete.push(locationKey);
   }

   for (const locationName of newLocationSet.values()) {
      const locationKey = createKey({ deployUnit: duId, location: locationName });

      if (type === FormChangeType.Remove) {
         action.delete.push(locationKey);
      } else {
         const { podCount } = newLocations[locationName];
         action.write[locationKey] = { value: podCount ?? 0 };
      }
   }
   return action;
}

export function useStageFormPodCount(stage: Stage, isNewQuota: boolean) {
   const initStagePodCountStore = useMemo(() => getStagePodCountStore(stage), [stage]);

   const oldStagePodCountStore = useMemo(() => (isNewQuota ? {} : initStagePodCountStore), [
      initStagePodCountStore,
      isNewQuota,
   ]);

   const [newStagePodCountStore, setNewStagePodCountStore] = useState(initStagePodCountStore);

   // сброс формы
   useEffect(() => {
      setNewStagePodCountStore(initStagePodCountStore);
   }, [initStagePodCountStore]);

   const podCountChangeListener: FormChangeListener = useMemo(
      () => ({
         listenFields: {
            deployUnit: ['locations'],
         },
         onChange({ path, newValue, oldValue, formValues, type }) {
            if (path !== 'locations') {
               return;
            }
            const { id } = formValues;

            const storeAction = getStagePodCountStoreAction(id, oldValue, newValue, type);

            if (!isEmptyStoreAction(storeAction)) {
               setNewStagePodCountStore(store => updateStore(store, storeAction));
            }
         },
      }),
      [],
   );

   return {
      oldStagePodCountStore,
      newStagePodCountStore,
      podCountChangeListener,
   };
}
