import { useEffect, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { YpLocation } from '../../../models/api';
import { Stage } from '../../../models/ui';
import { QuotaRequestKeys, QuotaKeyParams } from '../../../models/ui/quota';
import {
   getAllQuotaResources,
   getSidecarsQuotaResources,
   getUserQuotaResources,
   ResourceContextValue,
} from '../../../models/ui/stage/resources';
import { getQuota, RootState, selectQuotaTree } from '../../../redux';
import { useSidecarsQuota } from '../../../redux/hooks/useSidecarsQuota';
import { Tree, useArrayMemo } from '../../../utils';
import { useStageFormDiskType } from './useStageFormDiskType';
import { useStageFormPatcherRevisions } from './useStageFormPatcherRevisions';
import { useStageFormPodCount } from './useStageFormPodCount';
import { useStageFormResources } from './useStageFormResources';
import { useStageFormSidecars } from './useStageFormSidecars';
import { useStageFormUsedClusters } from './useStageFormUsedClusters';

interface Props {
   isNew: boolean;
   abc: string | null;
   initialAbc: string | null;
   stage: Stage;
}

export function useStageFormQuota({ isNew, abc, initialAbc, stage }: Props) {
   const isNewQuota = isNew || initialAbc !== abc;

   const { oldStagePodCountStore, newStagePodCountStore, podCountChangeListener } = useStageFormPodCount(
      stage,
      isNewQuota,
   );

   const { oldStageResourcesStore, newStageResourcesStore, resourcesChangeListener } = useStageFormResources(
      stage,
      isNewQuota,
   );

   const { oldStageSidecarsStore, newStageSidecarsStore, sidecarsChangeListener } = useStageFormSidecars(
      stage,
      isNewQuota,
   );

   const {
      oldStagePatcherRevisions,
      newStagePatcherRevisions,
      patcherRevisionsChangeListener,
   } = useStageFormPatcherRevisions(stage, isNewQuota);

   const { oldStageDiskTypeStore, newStageDiskTypeStore, diskTypeChangeListener } = useStageFormDiskType(
      stage,
      isNewQuota,
   );

   const quotaChangeListeners = useMemo(
      () => [
         podCountChangeListener,
         resourcesChangeListener,
         sidecarsChangeListener,
         patcherRevisionsChangeListener,
         diskTypeChangeListener,
      ],
      [
         patcherRevisionsChangeListener,
         podCountChangeListener,
         resourcesChangeListener,
         sidecarsChangeListener,
         diskTypeChangeListener,
      ],
   );

   const oldUserResourceValues = useMemo(() => getUserQuotaResources(oldStageResourcesStore, oldStagePodCountStore), [
      oldStagePodCountStore,
      oldStageResourcesStore,
   ]);

   const defaultSidecarQuota = useSidecarsQuota();

   const oldSidecarResourceValues = useMemo(
      () =>
         getSidecarsQuotaResources(
            oldStagePodCountStore,
            oldStageSidecarsStore,
            defaultSidecarQuota,
            oldStagePatcherRevisions,
            oldStageDiskTypeStore,
         ),
      [
         defaultSidecarQuota,
         oldStagePatcherRevisions,
         oldStagePodCountStore,
         oldStageSidecarsStore,
         oldStageDiskTypeStore,
      ],
   );

   const newSidecarResourceValues = useMemo(
      () =>
         getSidecarsQuotaResources(
            newStagePodCountStore,
            newStageSidecarsStore,
            defaultSidecarQuota,
            newStagePatcherRevisions,
            newStageDiskTypeStore,
         ),
      [
         defaultSidecarQuota,
         newStagePatcherRevisions,
         newStagePodCountStore,
         newStageSidecarsStore,
         newStageDiskTypeStore,
      ],
   );

   const newUserResourceValues = useMemo(() => getUserQuotaResources(newStageResourcesStore, newStagePodCountStore), [
      newStagePodCountStore,
      newStageResourcesStore,
   ]);

   const oldResourceValues = useMemo(() => getAllQuotaResources(oldUserResourceValues, oldSidecarResourceValues), [
      oldSidecarResourceValues,
      oldUserResourceValues,
   ]);

   const newResourceValues = useMemo(() => getAllQuotaResources(newUserResourceValues, newSidecarResourceValues), [
      newSidecarResourceValues,
      newUserResourceValues,
   ]);

   const resourceContextValue: ResourceContextValue = useMemo(
      () => ({
         stageResourcesStore: newStageResourcesStore,
         stagePodCountStore: newStagePodCountStore,
         stageSidecarsStore: newStageSidecarsStore,
         stageDiskTypeStore: newStageDiskTypeStore,
         stagePatcherRevisions: newStagePatcherRevisions,
      }),
      [
         newStageDiskTypeStore,
         newStagePatcherRevisions,
         newStagePodCountStore,
         newStageResourcesStore,
         newStageSidecarsStore,
      ],
   );

   // все отмеченные локации для подкачки квоты
   const usedAllClusters = useStageFormUsedClusters(newStagePodCountStore);

   // локации, где есть поды
   const usedNewClusters = useStageFormUsedClusters(newStagePodCountStore, false);

   // локации, где были поды
   const usedOldClusters = useStageFormUsedClusters(oldStagePodCountStore, false);

   // локации, которые были изменены
   // для новой квоты отсутствуют
   const usedClusters = useMemo(() => [...new Set([...usedNewClusters, ...(isNewQuota ? [] : usedOldClusters)])], [
      isNewQuota,
      usedNewClusters,
      usedOldClusters,
   ]);

   const dispatch = useDispatch();

   useEffect(() => {
      if (abc) {
         for (const cluster of usedAllClusters) {
            dispatch(getQuota.withRequestKey(QuotaRequestKeys.getOne({ cluster, abc }))(abc, cluster as YpLocation));
         }
      }
   }, [abc, dispatch, usedAllClusters]);

   const quotaOrder: (keyof QuotaKeyParams)[] = ['segment', 'abc', 'resourceGroup', 'resource', 'location', 'metric'];
   const quotaMemoOrder = (useArrayMemo(quotaOrder as any) as any) as typeof quotaOrder;
   const quota = (useSelector((state: RootState) => selectQuotaTree(state, quotaMemoOrder as any)) as any) as Tree<
      ['segment', 'abc', 'resourceGroup', 'resource', 'location', 'metric'],
      number
   >;
   const currentQuota = abc ? quota?.default?.[abc] : null;

   return {
      quotaChangeListeners,
      oldResourceValues,
      newResourceValues,
      usedClusters,
      currentQuota,
      resourceContextValue,
   };
}
