import { getSetDifference } from '@yandex-infracloud-ui/libs';
import { useEffect, useMemo, useState } from 'react';
import { DeployUnitDisk, DeployUnitNetworkBandwidth, DiskType, duPaths, Stage } from '../../../models/ui';
import { getStageDiskResources, getStageResourcesStore } from '../../../models/ui/stage/resources';
import { ResourceGroup } from '../../../modules/resources/config';
import { createKey, formName, getEmptyStoreAction, isEmptyStoreAction, StoreAction, updateStore } from '../../../utils';
import { FormChangeListener, FormChangeType } from '../../huge-form';

function getStageResourcesStoreAction(
   duId: string,
   path: string,
   oldValue: unknown,
   newValue: unknown,
   type: FormChangeType,
): StoreAction<number> {
   const action = getEmptyStoreAction<number>();

   const createDuKey = (data: Record<string, string>) => createKey({ deployUnit: duId, ...data });

   switch (path) {
      case formName(duPaths.cpu): {
         const key = createDuKey({ resourceGroup: ResourceGroup.Cpu, resource: '' });
         if (type === FormChangeType.Remove) {
            action.delete.push(key);
         } else {
            action.write[key] = { value: (newValue ?? 0) as number };
         }
         break;
      }
      case formName(duPaths.ram): {
         const key = createDuKey({ resourceGroup: ResourceGroup.Mem, resource: '' });
         if (type === FormChangeType.Remove) {
            action.delete.push(key);
         } else {
            action.write[key] = { value: (newValue ?? 0) as number };
         }
         break;
      }
      case formName(duPaths.networkBandwidth): {
         const key = createDuKey({ resourceGroup: ResourceGroup.NetworkBandwidth, resource: '' });
         if (type === FormChangeType.Remove) {
            action.delete.push(key);
         } else {
            action.write[key] = { value: ((newValue as DeployUnitNetworkBandwidth).guarantee ?? 0) as number };
         }
         break;
      }
      case formName(duPaths.disks): {
         const oldDiskResources = getStageDiskResources(oldValue as DeployUnitDisk[]);
         const newDiskResources = getStageDiskResources(newValue as DeployUnitDisk[]);

         const oldDiskTypesSet = new Set(Object.keys(oldDiskResources));
         const newDiskTypesSet = new Set(Object.keys(newDiskResources));

         const { removed } = getSetDifference(oldDiskTypesSet, newDiskTypesSet);
         for (const diskType of removed.values()) {
            action.delete.push(createDuKey({ resourceGroup: ResourceGroup.Disk, resource: diskType }));
            action.delete.push(createDuKey({ resourceGroup: ResourceGroup.DiskBandwidth, resource: diskType }));
         }

         for (const diskType of Object.keys(newDiskResources)) {
            const { disk = 0, diskBandwidth = 0 } = newDiskResources[diskType as DiskType] ?? {};

            const diskKey = createDuKey({ resourceGroup: ResourceGroup.Disk, resource: diskType });
            const diskBandwidthKey = createDuKey({ resourceGroup: ResourceGroup.DiskBandwidth, resource: diskType });

            if (type === FormChangeType.Remove) {
               action.delete.push(diskKey);
               action.delete.push(diskBandwidthKey);
            } else {
               action.write[diskKey] = { value: disk };
               action.write[diskBandwidthKey] = { value: diskBandwidth };
            }
         }
         break;
      }
   }

   return action;
}

export function useStageFormResources(stage: Stage, isNewQuota: boolean) {
   const initStageResourcesStore = useMemo(() => getStageResourcesStore(stage), [stage]);

   const oldStageResourcesStore = useMemo(() => (isNewQuota ? {} : initStageResourcesStore), [
      isNewQuota,
      initStageResourcesStore,
   ]);

   const [newStageResourcesStore, setNewStageResourcesStore] = useState(initStageResourcesStore);

   // сброс формы
   useEffect(() => {
      setNewStageResourcesStore(initStageResourcesStore);
   }, [initStageResourcesStore]);

   const resourcesChangeListener: FormChangeListener = useMemo(
      () => ({
         listenFields: {
            deployUnit: ['cpu', 'ram', 'networkBandwidth', 'disks'],
         },
         onChange({ path, newValue, oldValue, formValues, type }) {
            const { id } = formValues;
            const storeAction = getStageResourcesStoreAction(id, path, oldValue, newValue, type);

            if (!isEmptyStoreAction(storeAction)) {
               setNewStageResourcesStore(store => updateStore(store, storeAction));
            }
         },
      }),
      [],
   );

   return {
      oldStageResourcesStore,
      newStageResourcesStore,
      resourcesChangeListener,
   };
}
