import { useEffect, useMemo, useState } from 'react';
import {
   BoxFormParams,
   boxPaths,
   DeployUnitFormParams,
   DiskType,
   duPaths,
   SidecarName,
   Stage,
   stageLevels,
   WorkloadFormParams,
   workloadPaths,
} from '../../../models/ui';
import { getVolumeDiskType, getStageSidecarsStore, StageSidecarQuotaParams } from '../../../models/ui/stage/resources';
import { createKey, formName, getEmptyStoreAction, isEmptyStoreAction, StoreAction, updateStore } from '../../../utils';
import { ResourceGroup } from '../../../modules/resources/config';
import { FormChangeListener, FormChangeType } from '../../huge-form';
import { getInitialIds, StageFormInitialIds } from '../../../models/ui/stage/form-utils';

const setCopy = <T>(s: Set<T> | undefined | null) => new Set(s ? [...s.values()] : []);

interface GetStageSidecarsStoreActionParams {
   levelId: string;
   levelKeyData: StageFormInitialIds;
   path: string;
   oldValue: unknown;
   newValue: unknown;
   customDiskType: DiskType | null;
   type: FormChangeType;
}

function getStageSidecarsStoreAction({
   levelId,
   levelKeyData,
   path,
   newValue,
   oldValue,
   customDiskType,
   type,
}: GetStageSidecarsStoreActionParams): StoreAction<StageSidecarQuotaParams> {
   const action = getEmptyStoreAction<StageSidecarQuotaParams>();

   const createLevelKey = (data: Record<string, string>) => createKey({ ...levelKeyData, ...data });

   switch (levelId) {
      case stageLevels.deployUnit.id: {
         switch (path) {
            // PodAgent
            // так как он для всех du по умолчанию, отслеживаем сам факт создания/удаления du
            case formName(duPaths.id): {
               const key = createLevelKey({ sidecar: SidecarName.PodAgent });
               if (type === FormChangeType.Add) {
                  const params: StageSidecarQuotaParams = {};
                  if (customDiskType) {
                     params.diskType = customDiskType;
                  }
                  action.write[key] = { value: params };
               } else if (type === FormChangeType.Remove) {
                  action.delete.push(key);
               }
               break;
            }
            // TVM
            case formName(duPaths.tvm): {
               const { enabled, diskType, cpuLimit, memoryLimit } = newValue as DeployUnitFormParams['tvm'];
               const key = createLevelKey({ sidecar: SidecarName.TVM });
               if (type !== FormChangeType.Remove && enabled) {
                  // восстанавливаем значения
                  const params: StageSidecarQuotaParams = { resources: {} };
                  if (diskType) {
                     params.diskType = diskType;
                  }
                  if (cpuLimit) {
                     params.resources![ResourceGroup.Cpu] = cpuLimit;
                  }
                  if (memoryLimit) {
                     params.resources![ResourceGroup.Mem] = memoryLimit;
                  }
                  action.write[key] = { value: params };
               } else {
                  action.delete.push(key);
               }
               break;
            }
         }
         break;
      }
      case stageLevels.box.id: {
         switch (path) {
            case formName(boxPaths.juggler.enabled): {
               const value = newValue as BoxFormParams['juggler']['enabled'];

               const key = createLevelKey({ sidecar: SidecarName.Juggler });
               if (type !== FormChangeType.Remove && value) {
                  action.write[key] = { value: {} };
               } else {
                  action.delete.push(key);
               }
               break;
            }
            case formName(boxPaths.logrotateConfig.rawConfig): {
               const value = newValue as BoxFormParams['logrotateConfig']['rawConfig'];
               const prevValue = oldValue as BoxFormParams['logrotateConfig']['rawConfig'];

               const key = createLevelKey({ sidecar: SidecarName.LogRotate });
               if (type !== FormChangeType.Remove && value) {
                  if (Boolean(value) !== Boolean(prevValue)) {
                     action.write[key] = { value: {} };
                  }
               } else {
                  action.delete.push(key);
               }
               break;
            }
            case formName(boxPaths.dynamicResources): {
               const value = newValue as BoxFormParams['dynamicResources'];
               const prevValue = oldValue as BoxFormParams['dynamicResources'];

               const key = createLevelKey({ sidecar: SidecarName.DynamicResource });

               if (type !== FormChangeType.Remove && value.length > 0) {
                  if (value.length !== prevValue.length) {
                     action.write[key] = { value: {} };
                  }
               } else {
                  action.delete.push(key);
               }
               break;
            }
         }
         break;
      }
      case stageLevels.workload.id: {
         switch (path) {
            case formName(workloadPaths.coredumpPolicy.enabled): {
               const value = newValue as WorkloadFormParams['coredumpPolicy']['enabled'];
               const key = createLevelKey({ sidecar: SidecarName.Coredump });

               if (type !== FormChangeType.Remove && value) {
                  const params: StageSidecarQuotaParams = {};

                  if (customDiskType) {
                     params.diskType = customDiskType;
                  }
                  action.write[key] = { value: params };
               } else {
                  action.delete.push(key);
               }
               break;
            }
            case formName(workloadPaths.logs): {
               const value = newValue as WorkloadFormParams['logs'];

               // ключ для записи в хранилище, квота на логброкер на уровне деплой юнита
               const key = createLevelKey({ sidecar: SidecarName.Logbroker, box: '', workload: '' });

               // ключ для текущего ворклоада для сохранения информации о сайдкарах логов
               const usageKey = createLevelKey({ sidecar: SidecarName.Logbroker });

               const params: StageSidecarQuotaParams = {};

               if (customDiskType) {
                  params.diskType = customDiskType;
               }

               // чтобы не мутировать старые значения, далее используется обертка для создания нового множества setCopy

               if (type !== FormChangeType.Remove && value) {
                  // при включении логов в любом ворклоаде общее состояние по деплой юниту не может быть отключено
                  // поэтому запись идёт в любом случае, но с учётом использования ворклоадов с логами
                  action.write[key] = {
                     getValue: oldV => ({
                        ...params,
                        usageKeys: new Set([...setCopy(oldV?.usageKeys), usageKey]),
                     }),
                  };
               } else {
                  // при выключении нужно учитывать предыдущее состояние
                  // удалять, только если выключенный ворклоад был последний с логами
                  action.delete.push({
                     key,
                     condition: oldV => oldV?.usageKeys?.size === 1,
                  });

                  // иначе обновлять счётчик
                  action.write[key] = {
                     condition: oldV => (oldV?.usageKeys?.size ?? 0) > 1,
                     getValue: oldV => {
                        const usageKeys = setCopy(oldV?.usageKeys);
                        usageKeys.delete(usageKey);
                        return {
                           ...params,
                           usageKeys,
                        };
                     },
                  };
               }
               break;
            }
         }
         break;
      }
   }

   return action;
}

export function useStageFormSidecars(stage: Stage, isNewQuota: boolean) {
   const initStageSidecarsStore = useMemo(() => getStageSidecarsStore(stage), [stage]);

   const oldStageSidecarsStore = useMemo(() => (isNewQuota ? {} : initStageSidecarsStore), [
      initStageSidecarsStore,
      isNewQuota,
   ]);

   const [newStageSidecarsStore, setNewStageSidecarsStore] = useState(initStageSidecarsStore);

   // сброс формы
   useEffect(() => {
      setNewStageSidecarsStore(initStageSidecarsStore);
   }, [initStageSidecarsStore]);

   const sidecarsChangeListener: FormChangeListener = useMemo(
      () => ({
         listenFields: {
            [stageLevels.deployUnit.id]: [formName(duPaths.tvm), formName(duPaths.id)],
            [stageLevels.box.id]: [
               formName(boxPaths.juggler.enabled),
               formName(boxPaths.logrotateConfig.rawConfig),
               formName(boxPaths.dynamicResources),
            ],
            [stageLevels.workload.id]: [formName(workloadPaths.coredumpPolicy.enabled), formName(workloadPaths.logs)],
         },
         onChange({ levelId, path, newValue, oldValue, formValues, parentForms, type }) {
            let customDiskType: DiskType | null = null;

            switch (levelId) {
               case stageLevels.deployUnit.id: {
                  switch (path) {
                     case formName(duPaths.id): {
                        customDiskType = (formValues as DeployUnitFormParams).podAgentSidecarDiskType;
                        break;
                     }
                  }
                  break;
               }
               case stageLevels.workload.id: {
                  const duForm = parentForms[1].formParams as DeployUnitFormParams;

                  switch (path) {
                     case formName(workloadPaths.logs): {
                        customDiskType = duForm.logbrokerSidecarDiskType;
                        break;
                     }
                     case formName(workloadPaths.coredumpPolicy.enabled): {
                        const values = formValues as WorkloadFormParams;
                        customDiskType = getVolumeDiskType(duForm.disks, values.coredumpPolicy.volumeId);
                        break;
                     }
                  }
               }
            }

            const levelKeyData = getInitialIds(formValues, parentForms, levelId);

            const storeAction = getStageSidecarsStoreAction({
               levelId,
               levelKeyData,
               path,
               oldValue,
               newValue,
               customDiskType,
               type,
            });

            if (!isEmptyStoreAction(storeAction)) {
               setNewStageSidecarsStore(store => updateStore(store, storeAction));
            }
         },
      }),
      [],
   );

   return {
      oldStageSidecarsStore,
      newStageSidecarsStore,
      sidecarsChangeListener,
   };
}
