/**
 * вынесен отдельно, так как использует levelConfigs
 * иначе цикл levelConfigs -> stage-huge-form/components -> stage-huge-form/hooks -> levelConfigs
 */

import type { MonacoEditorProps, MonacoEditorValidationInfo } from '@yandex-infracloud-ui/monaco-editor';

import yaml from 'js-yaml';
import React, { useCallback, useMemo, useState } from 'react';

import { Stage, StagePatcher } from '../../models/ui';
import { StagePatcherVisitor } from '../../models/ui/stage/StagePatcherVisitor';
import { TStage } from '../../proto-typings';
import { pickFields } from '../../utils';
import { HugeFormRef } from '../huge-form';
import { formsFromValue } from '../huge-form/hooks/useFormsFromValue';
import { configs } from './configs';
import { getStageValueFromYaml, StageHugeFormEditView } from './models';

const getStageYamlText = (rawStage: TStage) => yaml.dump(pickFields(rawStage, ['spec', 'meta', 'labels']));

const defaultVisitor = new StagePatcherVisitor({
   options: {},
   duConfirmations: {},
});

export function useEditView({
   initialValue,
   rawStage,
   hugeFormRef,
   latestRawStage,
   rootRoutePath,
   isNew,
}: {
   initialValue: Stage;
   rawStage: TStage;
   hugeFormRef: React.RefObject<HugeFormRef<Stage>>;
   latestRawStage?: TStage;
   rootRoutePath(): string;
   isNew: boolean;
}) {
   // режим представления для редактирования
   const [editView, setEditView] = useState(StageHugeFormEditView.Form);

   // значение для формы
   const [formValue, setFormValue] = useState(initialValue);

   // редактируемое значение стейджа
   const [rawValue, setRawValue] = useState(rawStage);

   // редактируемое значение стейджа в виде ямла
   const yamlRawValue = useMemo(() => getStageYamlText(rawValue), [rawValue]);

   // текст для редактора
   const [yamlText, setYamlText] = useState(yamlRawValue);

   // ошибка парсинга yaml
   const [parseYamlError, setParseYamlError] = useState<string | null>(null);

   // информация о валидации yaml
   const [validationInfo, setValidationInfo] = useState<MonacoEditorValidationInfo | null>(null);
   const onChangeValidate: Required<MonacoEditorProps>['onChangeValidate'] = useCallback(({ validationInfo: info }) => {
      setValidationInfo(info);
   }, []);

   const validationStatus = validationInfo?.summary?.summaryStatus ?? null;

   // наличие критичных ошибок (валидность синтаксиса и т.д.)
   const hasYamlErrors = validationStatus === 'error';

   // сброс всех значений к исходным
   const revertValues = useCallback(() => {
      setFormValue(initialValue);
      setRawValue(rawStage);
      setYamlText(getStageYamlText(rawStage));
      setParseYamlError(null);
      setValidationInfo(null);
   }, [initialValue, rawStage]);

   // последнее значение стейджа из формы
   const getCurrentStageValueFromForm = useCallback(() => hugeFormRef?.current?.getValue() ?? formValue, [
      formValue,
      hugeFormRef,
   ]);

   // последнее значение стейджа из текста
   const getCurrentStageValueFromRaw = useCallback(() => {
      const { value: newStageValue, rawValue: newRawValue } = getStageValueFromYaml(yamlText);
      // side effect
      if (newRawValue) {
         // автоматически обновляем значения
         setRawValue(oldValue => ({ ...oldValue, ...newRawValue }));
      }
      return newStageValue ?? formValue;
   }, [formValue, yamlText]);

   // последнее значение стейджа в зависимости от представления
   const getCurrentStageValue = useCallback(() => {
      if (editView === StageHugeFormEditView.Form) {
         return getCurrentStageValueFromForm();
      }
      return getCurrentStageValueFromRaw();
   }, [editView, getCurrentStageValueFromForm, getCurrentStageValueFromRaw]);

   // актуальные формы в зависимости от представления
   const getCurrentForms = useCallback(() => {
      if (editView === StageHugeFormEditView.Form) {
         // реальные формы на странице
         return hugeFormRef.current?.getForms() ?? [];
      }
      const currentValue = getCurrentStageValueFromRaw();

      // в режиме текста создаём виртуальные формы на лету
      return formsFromValue({ value: currentValue, levelConfigs: configs, rootRoutePath, isNew });
   }, [editView, getCurrentStageValueFromRaw, hugeFormRef, isNew, rootRoutePath]);

   // установка формы из текста
   const setFormFromRaw = useCallback(() => {
      if (yamlRawValue === yamlText) {
         // текст не поменялся, просто возвращаемся к форме
         return;
      }

      const { value: newFormValues, error, rawValue: newRawValue } = getStageValueFromYaml(yamlText);
      setParseYamlError(error);
      if (newFormValues && newRawValue) {
         setRawValue(oldValue => ({ ...oldValue, ...newRawValue }));
         // текущая форма полностью сбрасывается
         hugeFormRef.current?.resetCurrentForm();

         // новое значение выставляется, форма сама приведётся к нужному виду
         window.requestAnimationFrame(() => setFormValue(newFormValues));
      }
   }, [hugeFormRef, yamlRawValue, yamlText]);

   // установка текста из формы
   const setRawFromForm = useCallback(() => {
      // значение из формы
      const currentEditedValue = hugeFormRef?.current?.getValue() ?? formValue;

      // собираем новое значение стейджа
      const newRawValue = StagePatcher.prepareToSave({
         asIs: false,
         description: '',
         editedValue: currentEditedValue,
         increment: false,
         latestRawStage,
         rawStage: rawValue,
         visitor: defaultVisitor,
      });

      // обновляем значение стейджа
      setRawValue(newRawValue);

      // обновляем текст в редакторе
      const newYamlText = getStageYamlText(newRawValue);
      setYamlText(newYamlText);

      return { newRawValue };
   }, [formValue, hugeFormRef, latestRawStage, rawValue]);

   const toogleEditView = useCallback(
      (newValue: StageHugeFormEditView) => {
         const action = newValue === StageHugeFormEditView.Raw ? setRawFromForm : setFormFromRaw;
         action();
         setEditView(newValue);
      },
      [setFormFromRaw, setRawFromForm],
   );

   return {
      editView,
      toogleEditView,
      parseYamlError,
      hasYamlErrors,
      onChangeValidate,
      validationInfo,
      rawValue,
      formValue,
      yamlText,
      setYamlText,
      getCurrentStageValue,
      getCurrentForms,
      revertValues,
   };
}
