import { ExternalLink, isEmpty } from '@yandex-infracloud-ui/libs';
import React, { useEffect, useState } from 'react';

import { EXTERNAL_LINKS } from '../../../../models';
import { sandboxApi } from '../../../../services';

interface UrlProps {
   url: string | null;
}

interface ResourceData {
   id: number | null;
   type: string | null;
   description: string | null;
}

const sbrRegExp = RegExp(/^sbr:[\d]+$/);

export const isSandboxUrl = (url: string | null) => {
   if (!url || isEmpty(url)) {
      return false;
   }

   if (sbrRegExp.test(url)) {
      return true;
   }

   return false;
};

export const SandboxUrl: React.FC<UrlProps> = React.memo(({ url }) => {
   const [resourceId, setResourceId] = useState<number | null>(null);
   const [resourceData, setResourceData] = useState<ResourceData | null>(null);

   useEffect(() => {
      let newResourceId = null;

      if (isSandboxUrl(url)) {
         const id = url?.split(':')[1];

         if (id && !isEmpty(id)) {
            newResourceId = Number(id);
         }
      }

      setResourceId(newResourceId);
   }, [url]);

   useEffect(() => {
      if (resourceId) {
         sandboxApi.getResource(resourceId).subscribe(
            (resource: ResourceData) => {
               const { type, description } = resource;

               setResourceData(
                  resource
                     ? {
                          id: resourceId,
                          type,
                          description,
                       }
                     : null,
               );
            },
            e => setResourceData(null),
         );
      } else {
         setResourceData(null);
      }
   }, [resourceId]);

   if (!resourceData) {
      return null;
   }

   return (
      <ExternalLink
         href={EXTERNAL_LINKS.sandboxResource(Number(resourceData.id))}
         title={resourceData.description || undefined}
      >
         {resourceData.type}
      </ExternalLink>
   );
});

SandboxUrl.displayName = 'SandboxUrl';
