import { Suggest } from '@yandex-data-ui/common';
import { classNames, EMPTY_VALUE, FieldLayout2, hasIncluded, useDismounted } from '@yandex-infracloud-ui/libs';
import { getIn, useFormikContext } from 'formik';
import React, { useCallback, useEffect, useState } from 'react';
import { firstValueFrom } from 'rxjs';
import { map, takeUntil } from 'rxjs/operators';

import { yasmApi } from '../../../../services';

import classes from './ITypeSubForm.module.css';

function renderItem(item: string) {
   return (
      <div key={item} className={classes.item}>
         {item}
      </div>
   );
}

interface Props {
   name: string;
   label: string;
   readonly: boolean;
   disabled: boolean;
}

export const ITypeSubForm: React.FC<Props> = React.memo(props => {
   const { name, readonly, disabled } = props;
   const form = useFormikContext();
   const value: string = getIn(form.values, name);

   const [query, setQuery] = useState(value ?? '');

   const dismounted = useDismounted();

   useEffect(() => {
      setQuery(value ?? '');
   }, [value]);

   const getITypes = useCallback(
      (q: string) =>
         firstValueFrom(
            yasmApi.getITypes().pipe(
               map(resp => resp.filter(item => hasIncluded(q, item))),
               takeUntil(dismounted),
            ),
         ).then(x => x ?? []),
      [dismounted],
   );

   const handleItemClick = useCallback(
      (item: string) => {
         setQuery(item);
         form.setFieldValue(name, item);
      },
      [form, name],
   );

   const handleQueryChange = useCallback(
      (v: string) => {
         setQuery(v);
         form.setFieldValue(name, v);
      },
      [form, name],
   );

   return (
      <div data-e2e={'YasmTags:Itype'}>
         <FieldLayout2 {...props} readonlyDots={readonly}>
            {readonly ? (
               <div className={classNames({ [classes.readonly]: readonly })}>{query || EMPTY_VALUE}</div>
            ) : (
               <Suggest<string>
                  debounce={150}
                  disabled={disabled}
                  getItems={getITypes}
                  getItemsOnMount={true}
                  itemHeight={35}
                  onItemClick={handleItemClick}
                  onUpdate={handleQueryChange}
                  placeholder={'Type to search...'}
                  renderItem={renderItem}
                  showItemsOnNoText={true}
                  text={query}
                  virtualized={true}
               />
            )}
         </FieldLayout2>
      </div>
   );
});

ITypeSubForm.displayName = 'ITypeSubForm';
