// import React, { useCallback, useEffect, useState } from 'react';

import { Suggest } from '@yandex-data-ui/common';
import { EMPTY_VALUE, hasIncluded, RemoveButton, useDismounted } from '@yandex-infracloud-ui/libs';
import { getIn, useFormikContext } from 'formik';
import React, { useCallback, useEffect, useState } from 'react';
import { firstValueFrom } from 'rxjs';
import { map, takeUntil } from 'rxjs/operators';

import { YasmTag } from '../../../../models/ui';
import { yasmApi } from '../../../../services';

import classes from './TagSubForm.module.css';

interface Props {
   index: number;
   name: string;
   itype: string | null;
   readonly: boolean;
   disabled: boolean;

   onRemove(i: number): void;
}

export const TagSubForm: React.FC<Props> = React.memo(({ name, index, readonly, disabled, onRemove, itype }) => {
   const form = useFormikContext();
   const value: YasmTag = getIn(form.values, name);

   const [keyQuery, setKeyQuery] = useState(value.key ?? '');
   const [valueQuery, setValueQuery] = useState(value.value ?? '');

   const dismounted = useDismounted();

   useEffect(() => {
      setKeyQuery(value.key ?? '');
   }, [value.key]);

   useEffect(() => {
      setValueQuery(value.value ?? '');
   }, [value.value]);

   const getTagKeys = useCallback(
      (query: string) =>
         itype
            ? firstValueFrom(
                 yasmApi.getTagKeys(itype).pipe(
                    map(resp => resp.filter(item => hasIncluded(query, item))),
                    takeUntil(dismounted),
                 ),
              ).then(x => x ?? [])
            : [],
      [itype, dismounted],
   );

   const getTags = useCallback(
      (query: string) =>
         itype && keyQuery
            ? firstValueFrom(
                 yasmApi.getTags(itype, keyQuery).pipe(
                    map(resp => resp.filter(item => hasIncluded(query, item))),
                    takeUntil(dismounted),
                 ),
              ).then(x => x ?? [])
            : [],
      [keyQuery, itype, dismounted],
   );

   const handleKeyQueryChange = useCallback(
      query => {
         form.setFieldValue(`${name}.key`, query);
         setKeyQuery(query);
         setValueQuery('');
      },
      [form, name],
   );

   const handleKeyItemClick = useCallback(
      (item: string) => {
         if (value.key !== item) {
            setValueQuery('');
            form.setFieldValue(`${name}.key`, item);
         }
         setKeyQuery(item);
      },
      [form, name, value.key],
   );

   const handleValueQueryChange = useCallback(
      query => {
         form.setFieldValue(`${name}.value`, query);
         setValueQuery(query);
      },
      [form, name],
   );

   const handleItemClick = useCallback(
      (item: string) => {
         setValueQuery(item);
         form.setFieldValue(`${name}.value`, item);
      },
      [form, name],
   );

   const handleRemove = useCallback(() => {
      onRemove(index);
   }, [index, onRemove]);

   return (
      <>
         <div>Tag name</div>

         <div data-e2e={'YasmTag:Key'}>
            {readonly ? (
               value.key ?? EMPTY_VALUE
            ) : (
               <Suggest<string>
                  getItems={getTagKeys}
                  renderItem={item => <div className={classes.item}>{item}</div>}
                  onItemClick={handleKeyItemClick}
                  onUpdate={handleKeyQueryChange}
                  placeholder={'Type to search...'}
                  debounce={250}
                  disabled={disabled || readonly}
                  text={keyQuery}
                  showItemsOnNoText={true}
                  getItemsOnMount={true}
               />
            )}
         </div>

         <div>Tag Value</div>

         <div data-e2e={'YasmTag:Value'}>
            {readonly ? (
               value.value ?? EMPTY_VALUE
            ) : (
               <Suggest<string>
                  key={`key-${itype}-${keyQuery}`}
                  getItems={getTags}
                  renderItem={item => <div className={classes.item}>{item}</div>}
                  onItemClick={handleItemClick}
                  onUpdate={handleValueQueryChange}
                  placeholder={'Type to search...'}
                  debounce={250}
                  disabled={disabled || readonly}
                  text={valueQuery}
                  showItemsOnNoText={true}
                  getItemsOnMount={true}
               />
            )}
         </div>

         {disabled || readonly ? null : (
            <div data-e2e={'YasmTag:RemoveTag'}>
               <RemoveButton onClick={handleRemove} />
            </div>
         )}
      </>
   );
});
