import { BytesField, EnumField2, FieldLayout2, InputField2, WarningPlate } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useContext } from 'react';

import { useSelector } from 'react-redux';

import { Box, BoxFormParams, boxPaths, DeployUnitDisk, resolvConfOptions, Workload } from '../../../../../models/ui';
import { SecretsReloadContext } from '../../../../../models/ui/secrets';

import { EnvSubForm } from '../../../../../modules/environment/components/EnvSubForm/EnvSubForm';
import { SecretsContext } from '../../../../../modules/secrets';
import { EResolvConf } from '../../../../../proto-typings';
import type { RootState } from '../../../../../redux';
import { formName } from '../../../../../utils';
import {
   CpuLimitField,
   EnabledSwitcherField,
   FormTab,
   FormTabsWithUrl,
   selectCurrentFormState,
} from '../../../../forms';
import { SubFormProps } from '../../../../huge-form';
import { useSecretsContextValueFromForm } from '../../../_common/useSecretsContextValueFromForm';
import { editOnlyForNewProps } from '../../../_common/utils';

import { BoxLayersSubForm } from '../../fields/BoxLayersSubForm/BoxLayersSubForm';
import { BoxStaticResourcesSubForm } from '../../fields/BoxStaticResourcesSubForm/BoxStaticResourcesSubForm';
import { BoxVolumesSubForm } from '../../fields/BoxVolumesSubForm/BoxVolumesSubForm';
import { CgroupFsSubForm } from '../../fields/CgroupFsSubForm/CgroupFsSubForm';
import { DockerSubForm } from '../DockerSubForm/DockerSubForm';
import { DynamicResourcesSubForm } from '../../fields/DynamicResourcesSubForm/DynamicResourcesSubForm';
import { JugglerAgentSubForm } from '../JugglerAgentSubForm/JugglerAgentSubForm';
import { LogrotateConfigSubForm } from '../../fields/LogrotateConfigSubForm/LogrotateConfigSubForm';
import { RootFsSettingsSubForm } from '../../fields/RootFsSettingsSubForm/RootFsSettingsSubForm';

import {
   BindSkynetHint,
   DynamicResourcesHint,
   EnvironmentHint,
   JugglerAgentHint,
   LogrotateConfigHint,
   ResolvConfHint,
   ResourceLimitHint,
} from '../../hints';

import classes from './BoxForm.module.css';

export const BoxForm: React.FC<SubFormProps<BoxFormParams, Box, Workload>> = ({
   disabled,
   form,
   formik,
   isRootEntityNew,
   readonly,
}) => {
   const reloadSecretUsages = useContext(SecretsReloadContext);

   const { id: deployUnitFormId, formParams: deployUnitFormParams } = form.parentForms[0];
   const { id: stageFormId, formParams: stageFormParams } = form.parentForms[1];

   const parentFormValuesSelector = useCallback((s: RootState) => selectCurrentFormState(s, deployUnitFormId)?.values, [
      deployUnitFormId,
   ]);

   const grandParentFormValuesSelector = useCallback((s: RootState) => selectCurrentFormState(s, stageFormId)?.values, [
      stageFormId,
   ]);

   const parentFormValues = useSelector(parentFormValuesSelector);
   const grandParentFormValues = useSelector(grandParentFormValuesSelector);

   const actualStageId = grandParentFormValues?.id ?? stageFormParams.id;

   const actualDeployUnitId = parentFormValues?.id ?? deployUnitFormParams.id;
   // mount volumes from any disks
   const actualDisks = parentFormValues?.disks ?? deployUnitFormParams.disks;
   // mount layers, static resources only from box.virtualDiskIdRef
   const actualDisk =
      actualDisks.find(
         // eslint-disable-next-line no-underscore-dangle
         (v: DeployUnitDisk) => formik.values.virtualDiskIdRef && v.id === formik.values.virtualDiskIdRef,
      ) ?? actualDisks[0]; // если virtualDiskIdRef не указан, то считаем дефолтным первый (например, для новых боксов)

   const formElement = (
      <div className={classes.form}>
         <InputField2
            name={'id'}
            label={'Box ID'}
            {...editOnlyForNewProps(readonly, disabled, form.isAdded, isRootEntityNew)}
            readonlyDots={true}
         />

         <EnumField2
            name={'resolvConf'}
            controlProps={{ options: resolvConfOptions }}
            label={'resolv.conf'}
            disabled={disabled}
            readonly={readonly}
            hint={<ResolvConfHint />}
            readonlyDots={true}
         />

         {formik.values.resolvConf === EResolvConf.EResolvConf_NAT64 ? (
            <WarningPlate className={classes.warning}>
               NAT64 is being deprecated. You should use nat64_local instead.
            </WarningPlate>
         ) : null}

         <EnabledSwitcherField
            name={'bindSkynet'}
            label={'Bind skynet'}
            disabled={disabled}
            readonly={readonly}
            hint={<BindSkynetHint />}
            readonlyDots={true}
         />

         <CpuLimitField
            name={'cpuPerBox'}
            label={'CPU per Box'}
            disabled={disabled}
            readonly={readonly}
            hint={<ResourceLimitHint />}
            readonlyDots={true}
         />

         <BytesField
            name={'ramPerBox'}
            label={'RAM per Box'}
            disabled={disabled}
            readonly={readonly}
            hint={<ResourceLimitHint />}
            readonlyDots={true}
         />

         <BytesField
            name={'anonymousMemoryLimit'}
            label={'Anonymous Memory Limit'}
            disabled={disabled}
            readonly={readonly}
            hint={<ResourceLimitHint />}
            readonlyDots={true}
         />

         <InputField2
            name={'threadLimit'}
            label={'Thread limit'}
            disabled={disabled}
            readonly={readonly}
            hint={<ResourceLimitHint />}
            controlProps={{ controlProps: { min: 0 }, type: 'number' }}
            readonlyDots={true}
         />

         <FieldLayout2
            name={'logrotateConfig'}
            label={'Logrotate config'}
            hint={<LogrotateConfigHint />}
            bigLabel={true}
            hideErrors={true}
         >
            <LogrotateConfigSubForm name={'logrotateConfig'} disabled={disabled} readonly={readonly} form={form} />
         </FieldLayout2>

         <FieldLayout2 name={'environment'} label={'Environment variables'} hint={<EnvironmentHint />} bigLabel={true}>
            <EnvSubForm name={'environment'} readonly={readonly} />
         </FieldLayout2>

         <FieldLayout2
            name={'juggler'}
            label={'Juggler alerting'}
            hint={<JugglerAgentHint />}
            hideErrors={true}
            bigLabel={true}
         >
            <JugglerAgentSubForm name={'juggler'} disabled={disabled} readonly={readonly} form={form} />
         </FieldLayout2>

         <FieldLayout2
            name={formName(boxPaths.rootFsSettings)}
            label={'Rootfs settings'}
            hideErrors={true}
            bigLabel={true}
         >
            <RootFsSettingsSubForm
               name={formName(boxPaths.rootFsSettings)}
               disabled={disabled}
               readonly={readonly}
               form={form}
            />
         </FieldLayout2>

         <FieldLayout2
            name={formName(boxPaths.rootFsSettings)}
            label={'Cgroupfs settings'}
            hideErrors={true}
            bigLabel={true}
         >
            <CgroupFsSubForm
               name={formName(boxPaths.cgroupFsMountMode)}
               disabled={disabled}
               readonly={readonly}
               form={form}
            />
         </FieldLayout2>
      </div>
   );

   return (
      <SecretsContext.Provider value={useSecretsContextValueFromForm(form, reloadSecretUsages)}>
         <FormTabsWithUrl defaultTab={'form'}>
            <FormTab id={'form'} title={'Form'}>
               {formElement}
            </FormTab>
            <FormTab
               id={'resources'}
               title={'Resources'}
               promo={{ key: 'promo.visible.box.tabs.resources', message: 'Box resources are here now!' }}
            >
               <div className={classes.form}>
                  <InputField2
                     // eslint-disable-next-line no-underscore-dangle
                     name={formName(boxPaths.virtualDiskIdRef)}
                     label={'Virtual disk ID'}
                     {...editOnlyForNewProps(readonly, disabled, form.isAdded, isRootEntityNew)}
                     // TODO: для мультидисков тут нужен будет селект #DEPLOY-5373
                     placeholder={actualDisk.id}
                     readonlyDots={true}
                     disabled={true}
                     required={actualDisks.length > 1}
                  />

                  <DockerSubForm name={formName(boxPaths.dockerImage)} readonly={readonly} disabled={disabled} />

                  <BoxLayersSubForm
                     name={formName(boxPaths.layers)}
                     readonly={readonly}
                     disabled={disabled}
                     stageId={actualStageId}
                     duId={actualDeployUnitId}
                     disk={actualDisk}
                     dockerEnabled={formik.values?.dockerImage?.enabled === true}
                  />

                  <BoxStaticResourcesSubForm
                     name={'staticResources'}
                     disabled={disabled}
                     readonly={readonly}
                     stageId={actualStageId}
                     duId={actualDeployUnitId}
                     disk={actualDisk}
                  />

                  <FieldLayout2
                     name={'dynamicResources'}
                     label={'Dynamic resources'}
                     hint={<DynamicResourcesHint />}
                     hideErrors={true}
                     bigLabel={true}
                  >
                     <DynamicResourcesSubForm
                        name={'dynamicResources'}
                        disabled={disabled}
                        readonly={readonly}
                        dynamicResourcePrefix={`${actualStageId}.${actualDeployUnitId}`}
                        form={form}
                     />
                  </FieldLayout2>

                  <BoxVolumesSubForm
                     name={'volumes'}
                     stageId={actualStageId}
                     duId={actualDeployUnitId}
                     disks={actualDisks}
                     disabled={disabled}
                     readonly={readonly}
                  />
               </div>
            </FormTab>
         </FormTabsWithUrl>
      </SecretsContext.Provider>
   );
};
