import React, { useCallback } from 'react';

import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { FieldLayout2, FormButton, InputField2 } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';

import { Box, BoxFormParams, BoxJugglerSettings, SidecarName, Workload } from '../../../../../models/ui';
import { EnabledSwitcherField } from '../../../../forms';
import { JugglerBundleHint, JugglerPortHint } from '../../hints';
import { BundleRow } from './BundleRow';
import { SidecarResourcePlate } from '../../../../stage-huge-form/components';
import { SubForm } from '../../../../huge-form';
import { getInitialIds } from '../../../../../models/ui/stage/form-utils';

import classes from './JugglerAgentSubForm.module.css';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
   form?: SubForm<BoxFormParams, Box, Workload>;
}

export const JugglerAgentSubForm: React.FC<Props> = React.memo(({ readonly, disabled, name, form: subForm }) => {
   const form = useFormikContext();
   const { deployUnit: duId = '', box: boxId = '' } = subForm
      ? getInitialIds(subForm.formParams, subForm.parentForms, subForm.levelConfig.id)
      : {};

   const { value } = form.getFieldMeta<BoxJugglerSettings>(name);

   const handleAddBundle = useCallback(() => {
      const newBundles = [...value.bundles, { url: null }];

      form.setFieldValue(`${name}.bundles`, newBundles);
   }, [form, name, value]);

   const handleRemoveBundle = useCallback(
      (i: number) => {
         const newBundles = [...value.bundles];
         newBundles.splice(i, 1);

         form.setFieldTouched(`${name}.bundles[${i}]`, false);
         form.setFieldValue(`${name}.bundles`, newBundles);
      },
      [form, name, value],
   );

   return (
      <div data-test={'Box:JugglerSettings'} className={classes.wrapper}>
         {!readonly || !value?.enabled ? (
            <EnabledSwitcherField name={`${name}.enabled`} label={null} disabled={disabled} readonly={readonly} />
         ) : null}

         {value.enabled ? (
            <>
               <div className={classes.warning}>
                  <SidecarResourcePlate sidecar={SidecarName.Juggler} deployUnit={duId} box={boxId} />
               </div>

               <InputField2
                  name={`${name}.port`}
                  label={'Juggler agent port'}
                  hint={<JugglerPortHint />}
                  controlProps={{ controlProps: { min: 0 }, type: 'number' }}
                  disabled={disabled}
                  readonly={readonly}
                  readonlyDots={readonly}
               />

               {value.bundles?.length && value.bundles.length > 0 ? (
                  <div data-test={'JugglerSettings:Bundles'}>
                     <FieldLayout2
                        name={`${name}.bundles`}
                        label={'Juggler bundles'}
                        hideErrors={true}
                        hint={<JugglerBundleHint />}
                        readonly={readonly}
                        readonlyDots={readonly}
                     >
                        {value.bundles.map((_, i) => (
                           <BundleRow
                              // eslint-disable-next-line react/no-array-index-key
                              key={i}
                              index={i}
                              name={`${name}.bundles[${i}]`}
                              onRemove={handleRemoveBundle}
                              disabled={disabled}
                              readonly={readonly}
                           />
                        ))}
                     </FieldLayout2>
                  </div>
               ) : null}
            </>
         ) : null}

         {!(disabled || readonly) && value.enabled ? (
            <FormButton
               icon={faPlus}
               onClick={handleAddBundle}
               skipLeftSpace={true}
               dataE2e={'JugglerSettings:AddJugglerBundle'}
            >
               Add Juggler bundle
            </FormButton>
         ) : null}
      </div>
   );
});

JugglerAgentSubForm.displayName = 'JugglerAgentSubForm';
