/* eslint-disable no-underscore-dangle */
import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { Table, TableColumnConfig } from '@yandex-cloud/uikit';
import { YCSelectItem, YCSelectItemsGroup } from '@yandex-data-ui/common';
import { EMPTY_VALUE, FieldLayout2, FormButton, isEmpty, RemoveButton } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { ReactNode, useCallback, useMemo } from 'react';
import { Link } from 'react-router-dom';

import { urlBuilder } from '../../../../../models';
import { BoxLayer, DeployUnitDisk, DiskLayer, DiskLayerType } from '../../../../../models/ui';
import { getHexRef } from '../../../../../utils';
import { YCSelectField } from '../../../../forms';
import { SandboxUrl } from '../../../_common/SanboxUrlField/SandboxUrl';
import { BoxLayersHint } from '../../hints';

import classes from './BoxLayersSubForm.module.css';

interface Props {
   disabled: boolean;
   disk: DeployUnitDisk;
   dockerEnabled: boolean;
   duId: string;
   name: string;
   readonly: boolean;
   stageId: string;
}

interface TableRow {
   buttons: ReactNode;
   checksum: ReactNode;
   id: ReactNode;
   link: ReactNode;
   url: ReactNode;
}

const columns: TableColumnConfig<TableRow>[] = [
   { id: 'id', name: 'Layer ID' },
   { id: 'url', name: 'URL' },
   { id: 'link', name: 'Link' },
   { id: 'checksum', name: 'Checksum' },
   { id: 'buttons', name: '' },
];

export const BoxLayersSubForm: React.FC<Props> = React.memo(
   ({ disabled, name, readonly, dockerEnabled, stageId, duId, disk }) => {
      const form = useFormikContext();
      const { value } = form.getFieldMeta<BoxLayer[]>(name);

      const layersMap = useMemo(
         () => new Map<string, DiskLayer>((disk?.layers ?? []).map(layer => [layer._ref, layer])),
         [disk],
      );

      const LayerSelectOptions: YCSelectItem[] | YCSelectItemsGroup[] = useMemo(() => {
         if (!disk.id || !disk.layers || isEmpty(disk.layers)) {
            return [];
         }

         return [
            {
               groupTitle: disk.id,
               items: disk.layers.map(layer => ({
                  value: layer._ref,
                  title: layer.id || EMPTY_VALUE,
                  meta: layer.url ?? 'unknown type',
               })),
            },
         ];
      }, [disk]);

      const handleAddLayer = useCallback(() => {
         const newLayer: BoxLayer = {
            _layerRef: getHexRef(),
         };

         form.setFieldValue(name, [...value, newLayer]);
      }, [form, name, value]);

      const handleRemoveLayer = useCallback(
         (i: number) => {
            const newValue = [...value];

            newValue.splice(i, 1);

            form.setFieldTouched(`${name}[${i}]`, false);
            form.setFieldValue(name, newValue);
         },
         [form, name, value],
      );

      const getReadonlyLayerId = useCallback(
         (ref: string | null) => disk.layers.find(v => v._ref === ref)?.id || EMPTY_VALUE,
         [disk],
      );

      return (
         <FieldLayout2 name={name} label={'Layers'} hint={<BoxLayersHint />} hideErrors={true} bigLabel={true}>
            {/* TODO: hideErrors={!isEmpty(value)} DEPLOY-5620 */}
            <div data-test={'Box:Layers'}>
               {!readonly ? (
                  <div>
                     {isEmpty(disk.layers) ? 'You have no layers yet. ' : null}
                     You should create/edit disk layers{' '}
                     <Link to={urlBuilder.stageEditDeployUnitDisks(stageId, duId)}>here</Link>.
                  </div>
               ) : null}

               {value.length === 0 ? (
                  <>{readonly ? <div>{EMPTY_VALUE}</div> : null}</>
               ) : (
                  <>
                     <Table
                        className={classes.table}
                        columns={columns}
                        data={(value ?? []).map((layer, i) => ({
                           id: (
                              <div className={classes.id} data-test={`Layer:${i + 1}:Id`}>
                                 {readonly ? (
                                    getReadonlyLayerId(layer._layerRef)
                                 ) : (
                                    <YCSelectField
                                       type={'single'}
                                       name={`${name}[${i}]._layerRef`}
                                       label={null}
                                       required={true}
                                       readonly={readonly}
                                       readonlyDots={readonly}
                                       disabled={disabled || isEmpty(LayerSelectOptions)}
                                       controlProps={{
                                          items: LayerSelectOptions,
                                          placeholder: !isEmpty(LayerSelectOptions) ? 'Select layer' : undefined,
                                          showSearch: false,
                                          showItemMeta: true,
                                       }}
                                    />
                                 )}
                              </div>
                           ),
                           url: (
                              <div data-test={`Layer:${i + 1}:URL`}>
                                 {layer._layerRef ? layersMap.get(layer._layerRef)?.url ?? '' : ''}
                              </div>
                           ),
                           link: (
                              <div data-test={`Layer:${i + 1}:Link`}>
                                 {layer._layerRef && layersMap.get(layer._layerRef)?.type === DiskLayerType.Url ? (
                                    <SandboxUrl url={layersMap.get(layer._layerRef)?.url ?? null} />
                                 ) : (
                                    ''
                                 )}
                              </div>
                           ),
                           checksum: (
                              <div data-test={`Layer:${i + 1}:Checksum`}>
                                 {layer._layerRef ? layersMap.get(layer._layerRef)?.checksum ?? '' : ''}
                              </div>
                           ),
                           buttons:
                              !disabled && !readonly ? (
                                 <div data-test={`Layer:${i + 1}:Buttons`}>
                                    {dockerEnabled || value.length > 1 ? (
                                       <RemoveButton onClick={() => handleRemoveLayer(i)} />
                                    ) : null}
                                 </div>
                              ) : (
                                 ' '
                              ),
                        }))}
                     />
                  </>
               )}

               {disabled || readonly || isEmpty(disk.layers) ? null : (
                  <FormButton icon={faPlus} onClick={handleAddLayer} skipLeftSpace={true} dataE2e={'Layers:AddLayer'}>
                     Add layer
                  </FormButton>
               )}
            </div>
         </FieldLayout2>
      );
   },
);

BoxLayersSubForm.displayName = 'BoxLayersSubForm';
