import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { Table, TableColumnConfig } from '@yandex-cloud/uikit';
import { YCSelectItem, YCSelectItemsGroup } from '@yandex-data-ui/common';
import { EMPTY_VALUE, FieldLayout2, FormButton, InputField2, isEmpty, RemoveButton } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { ReactNode, useCallback, useMemo } from 'react';
import { Link } from 'react-router-dom';

import { urlBuilder } from '../../../../../models';
import { BoxStaticResource, DeployUnitDisk, DiskStaticResource, StaticResourceType } from '../../../../../models/ui';
import { getHexRef } from '../../../../../utils';
import { YCSelectField } from '../../../../forms';
import { BoxStaticResourcesHint } from '../../hints';

import classes from './BoxStaticResourcesSubForm.module.css';

interface Props {
   disabled: boolean;
   disk: DeployUnitDisk;
   duId: string;
   name: string;
   readonly: boolean;
   stageId: string;
}

interface TableRow {
   buttons: ReactNode;
   id: ReactNode;
   type: ReactNode;
   mountPoint: ReactNode;
}

const columns: TableColumnConfig<TableRow>[] = [
   { id: 'id', name: 'Static Resource ID' },
   { id: 'type', name: 'Type' },
   { id: 'mountPoint', name: 'Mount point' },
   { id: 'buttons', name: '' },
];

export const BoxStaticResourcesSubForm: React.FC<Props> = React.memo(
   ({ disabled, name, readonly, stageId, duId, disk }) => {
      const form = useFormikContext();
      const { value } = form.getFieldMeta<BoxStaticResource[]>(name);

      const staticResourcesMap = useMemo(
         () =>
            new Map<string, DiskStaticResource>(
               // eslint-disable-next-line no-underscore-dangle
               (disk?.staticResources ?? []).map(staticResource => [staticResource._ref, staticResource]),
            ),
         [disk],
      );

      const StaticResourceSelectOptions: YCSelectItem[] | YCSelectItemsGroup[] = useMemo(() => {
         if (!disk.id || !disk.staticResources || isEmpty(disk.staticResources)) {
            return [];
         }

         return [
            {
               groupTitle: disk.id,
               items: disk.staticResources.map(v => ({
                  // eslint-disable-next-line no-underscore-dangle
                  value: v._ref,
                  title: v.id || EMPTY_VALUE,
                  meta:
                     v.type === StaticResourceType.Url && v.url
                        ? v.url
                        : v.type === StaticResourceType.Files
                        ? 'files'
                        : 'unknown type',
               })),
            },
         ];
      }, [disk]);

      const handleAddStaticResource = useCallback(() => {
         const newStaticResource: BoxStaticResource = {
            mountPoint: null,
            _staticResourceRef: getHexRef(),
         };

         form.setFieldValue(name, [...value, newStaticResource]);
      }, [form, name, value]);

      const handleRemoveStaticResource = useCallback(
         (i: number) => {
            const newValue = [...value];

            newValue.splice(i, 1);

            form.setFieldTouched(`${name}[${i}]`, false);
            form.setFieldValue(name, newValue);
         },
         [form, name, value],
      );

      const getReadonlyStaticResourceId = useCallback(
         // eslint-disable-next-line no-underscore-dangle
         (ref: string | null) => disk.staticResources.find(v => v._ref === ref)?.id || EMPTY_VALUE,
         [disk],
      );

      return (
         <FieldLayout2
            name={name}
            label={'Mount static resources'}
            hint={<BoxStaticResourcesHint />}
            hideErrors={true}
            bigLabel={true}
         >
            <div data-test={'Box:StaticResources'}>
               {!readonly ? (
                  <div>
                     {isEmpty(disk.staticResources) ? 'You have no static resources yet. ' : null}
                     You should create/edit disk static resources{' '}
                     <Link to={urlBuilder.stageEditDeployUnitDisks(stageId, duId)}>here</Link>.
                  </div>
               ) : null}

               {value.length === 0 ? (
                  <>{readonly ? <div>{EMPTY_VALUE}</div> : null}</>
               ) : (
                  <>
                     <Table
                        className={classes.table}
                        columns={columns}
                        data={(value ?? []).map((staticResource, i) => {
                           // eslint-disable-next-line no-underscore-dangle
                           const staticResourceRef = staticResource._staticResourceRef;
                           return {
                              id: (
                                 <div className={classes.id} data-test={`StaticResource:${i + 1}:Id`}>
                                    {readonly ? (
                                       getReadonlyStaticResourceId(staticResourceRef)
                                    ) : (
                                       <YCSelectField
                                          type={'single'}
                                          name={`${name}[${i}]._staticResourceRef`}
                                          label={null}
                                          required={true}
                                          readonly={readonly}
                                          readonlyDots={readonly}
                                          disabled={disabled || isEmpty(StaticResourceSelectOptions)}
                                          controlProps={{
                                             items: StaticResourceSelectOptions,
                                             placeholder: !isEmpty(StaticResourceSelectOptions)
                                                ? 'Select static resource'
                                                : undefined,
                                             showSearch: false,
                                             showItemMeta: true,
                                          }}
                                       />
                                    )}
                                 </div>
                              ),
                              type: (
                                 <div data-test={`StaticResource:${i + 1}:Type`}>
                                    {staticResourceRef ? staticResourcesMap.get(staticResourceRef)?.type ?? '' : ''}
                                 </div>
                              ),
                              mountPoint: (
                                 <div className={classes.mountPoint} data-test={`StaticResource:${i + 1}:MountPoint`}>
                                    <InputField2
                                       name={`${name}[${i}].mountPoint`}
                                       label={null}
                                       disabled={disabled}
                                       readonly={readonly}
                                       readonlyDots={readonly}
                                       placeholder={'Mount point directory'}
                                    />
                                 </div>
                              ),
                              buttons:
                                 !disabled && !readonly ? (
                                    <div className={classes.buttons} data-test={`StaticResource:${i + 1}:Buttons`}>
                                       <RemoveButton onClick={() => handleRemoveStaticResource(i)} />
                                    </div>
                                 ) : (
                                    ' '
                                 ),
                           };
                        })}
                     />
                  </>
               )}

               {disabled || readonly || isEmpty(disk.staticResources) ? null : (
                  <FormButton
                     icon={faPlus}
                     onClick={handleAddStaticResource}
                     skipLeftSpace={true}
                     dataE2e={'StaticResources:AddStaticResource'}
                  >
                     Add static resource
                  </FormButton>
               )}
            </div>
         </FieldLayout2>
      );
   },
);

BoxStaticResourcesSubForm.displayName = 'BoxStaticResourcesSubForm';
