import { YCSelectItem, YCSelectItemsGroup } from '@yandex-data-ui/common';
import {
   EMPTY_VALUE,
   EnumField2,
   IEnumOption,
   InputField2,
   isEmpty,
   RemoveButton,
   unique,
} from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback, useMemo } from 'react';
import { Link } from 'react-router-dom';

import { urlBuilder } from '../../../../../models';
import { BoxVolume, DeployUnitDisk, VolumeMountMode } from '../../../../../models/ui';
import { FormSubSection, YCSelectField } from '../../../../forms';

import classes from './BoxVolumeCard.module.css';

const typeOptions: IEnumOption[] = [
   { title: VolumeMountMode.ReadOnly, value: VolumeMountMode.ReadOnly },
   { title: VolumeMountMode.ReadWrite, value: VolumeMountMode.ReadWrite },
];

interface Props {
   name: string;
   readonly: boolean;
   disabled: boolean;
   index: number;

   stageId: string;
   duId: string;
   disks: DeployUnitDisk[];

   onRemove: (i: number) => void;
}

export const BoxVolumeCard: React.FC<Props> = React.memo(
   ({ name, readonly, disabled, index, stageId, duId, disks, onRemove }) => {
      const form = useFormikContext();
      const { value } = form.getFieldMeta<BoxVolume>(name);

      const VolumeSelectOptions: YCSelectItem[] | YCSelectItemsGroup[] = useMemo(() => {
         const volumes = (disks ?? [])
            .flatMap(v => v.volumes ?? [])
            .map(v => v.id)
            .filter(v => !isEmpty(v));

         // рисуем красивый select с группировкой по дискам, если все volumes уникальные
         if (!isEmpty(volumes) && volumes.length === unique(volumes).length) {
            return (disks ?? []).map(d => ({
               groupTitle: d.id,
               items: (d.volumes ?? []).map(volume => ({
                  // eslint-disable-next-line no-underscore-dangle
                  value: volume._ref,
                  title: volume.removed ? `${volume.id} (removed)` : volume.id || EMPTY_VALUE,
               })),
            }));
         }

         // рисуем некрасивый, если пользователь зачем-то добавил дубликатов
         // YP не даст ему катить кривую спеку,
         // но нам тоже надо красиво выходить такого из положения
         return unique(volumes).map(v => ({
            value: v,
            title: v,
         }));
      }, [disks]);

      const getReadonlyVolumeId = useCallback(
         // eslint-disable-next-line no-underscore-dangle
         () => disks.flatMap(v => v.volumes).find(v => v._ref === value._volumeRef)?.id || EMPTY_VALUE,
         [disks, value],
      );

      return (
         <div data-test={`Volume:${index + 1}`}>
            <FormSubSection title={getReadonlyVolumeId()}>
               {!readonly && !disabled ? (
                  <div className={classes.removeButton} data-e2e={'Volume:Remove'}>
                     <RemoveButton onClick={() => onRemove(index)} size={'lg'} />
                  </div>
               ) : null}

               {!readonly && (
                  <div data-test={'Volume:Id'}>
                     <YCSelectField
                        type={'single'}
                        name={`${name}._volumeRef`}
                        label={'Volume ID'}
                        required={true}
                        readonly={readonly}
                        readonlyDots={readonly}
                        disabled={disabled || isEmpty(VolumeSelectOptions)}
                        help={
                           !readonly ? (
                              <>
                                 {isEmpty(VolumeSelectOptions) ? 'You have no volumes yet. ' : null}
                                 {'You should create/edit disk volumes '}
                                 <Link to={urlBuilder.stageEditDeployUnitDisks(stageId, duId)}>here</Link>.
                              </>
                           ) : null
                        }
                        controlProps={{
                           items: VolumeSelectOptions,
                           placeholder: !isEmpty(VolumeSelectOptions) ? 'Select volume' : undefined,
                           showSearch: false,
                        }}
                     />
                  </div>
               )}

               <div data-test={'Volume:Mode'}>
                  <EnumField2
                     name={`${name}.mode`}
                     label={'Mode'}
                     required={true}
                     disabled={disabled}
                     readonly={readonly}
                     controlProps={{ options: typeOptions }}
                     readonlyDots={readonly}
                  />
               </div>

               <div data-test={'Volume:MountPoint'}>
                  <InputField2
                     name={`${name}.mountPoint`}
                     label={'Mount point'}
                     required={true}
                     readonly={readonly}
                     readonlyDots={readonly}
                     disabled={disabled}
                  />
               </div>
            </FormSubSection>
         </div>
      );
   },
);

BoxVolumeCard.displayName = 'BoxVolumeCard';
