import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { EMPTY_VALUE, FieldLayout2, FormButton } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback } from 'react';

import { BoxVolume, DeployUnitDisk, VolumeMountMode } from '../../../../../models/ui';
import { VolumeHint } from '../../hints';
import { BoxVolumeCard } from './BoxVolumeCard';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;

   stageId: string;
   duId: string;
   disks: DeployUnitDisk[];
}

export const BoxVolumesSubForm: React.FC<Props> = React.memo(({ disabled, name, readonly, stageId, duId, disks }) => {
   const form = useFormikContext();
   const { value } = form.getFieldMeta<BoxVolume[]>(name);

   const handleAddVolume = useCallback(() => {
      const newVolume: BoxVolume = {
         mode: VolumeMountMode.ReadOnly,
         mountPoint: null,
         _volumeRef: null,
      };

      form.setFieldValue(name, [...value, newVolume]);
   }, [form, name, value]);

   const handleRemoveVolume = useCallback(
      (i: number) => {
         const newValue = [...value];

         newValue.splice(i, 1);

         form.setFieldTouched(`${name}[${i}]`, false);
         form.setFieldValue(name, newValue);
      },
      [form, name, value],
   );

   return (
      <FieldLayout2
         name={name}
         label={'Mount volumes'}
         disabled={disabled}
         readonly={readonly}
         bigLabel={true}
         hideErrors={true}
         hint={<VolumeHint />}
      >
         <div data-test={'Box:Volumes'}>
            {value.length === 0 ? (
               <>{readonly ? <div>{EMPTY_VALUE}</div> : null}</>
            ) : (
               value.map((_, i) => (
                  <BoxVolumeCard
                     name={`${name}[${i}]`}
                     // eslint-disable-next-line react/no-array-index-key
                     key={i}
                     index={i}
                     disabled={disabled}
                     readonly={readonly}
                     onRemove={handleRemoveVolume}
                     stageId={stageId}
                     duId={duId}
                     disks={disks}
                  />
               ))
            )}

            {disabled || readonly ? null : (
               <FormButton icon={faPlus} onClick={handleAddVolume} skipLeftSpace={true} dataE2e={'Volumes:AddVolume'}>
                  Add volume
               </FormButton>
            )}
         </div>
      </FieldLayout2>
   );
});

BoxVolumesSubForm.displayName = 'BoxVolumesSubForm';
