import React, { useCallback, useMemo } from 'react';

import {
   classNames,
   EMPTY_VALUE,
   ExternalLink,
   FieldLayout2,
   FormCard,
   InputField2,
   RemoveButton,
} from '@yandex-infracloud-ui/libs';
import { getIn, useFormikContext } from 'formik';

import { EXTERNAL_LINKS } from '../../../../../models';
import { BoxDynamicResource } from '../../../../../models/ui';
import { SandboxUrlField } from '../../../_common/SanboxUrlField/SandboxUrlField';
import { AdvancedSettingsSubForm } from './AdvancedSettingsSubForm';
import { NotifyPolicySubForm } from './NotifyPolicySubForm';

import classes from './DynamicResourceRecord.module.css';

interface Props {
   disabled: boolean;
   dynamicResourcePrefix: string;
   index: number;
   name: string;
   readonly: boolean;

   onRemove(i: number): void;
}

export const DynamicResourceRecord: React.FC<Props> = React.memo(
   ({ name, index, readonly, disabled, dynamicResourcePrefix, onRemove }) => {
      const form = useFormikContext();
      const value: BoxDynamicResource = getIn(form.values, name);

      const handleResourceRemove = useCallback(() => onRemove!(index!), [index, onRemove]);

      const handleUrlRemove = useCallback(
         (i: number) => {
            if (value.urls?.[i]) {
               const newUrls = [...value.urls];
               newUrls.splice(i, 1);

               form.setFieldValue(`${name}.urls`, newUrls);
            }
         },
         [form, name, value.urls],
      );

      const customSettings = useMemo(() => value.customSettings?.deployGroups || value.customSettings?.requiredLabels, [
         value,
      ]);

      return (
         <div className={classes.wrapper} data-test={`DynamicResource:${index + 1}`}>
            <FormCard
               title={value.id ? value.id : EMPTY_VALUE}
               hasRemoveButton={!customSettings && !readonly && !disabled}
               onRemove={handleResourceRemove}
               className={classNames(classes.card, { [classes.readonly]: readonly })}
            >
               <div data-test={'Resource:Id'}>
                  <InputField2
                     name={`${name}.id`}
                     label={'Resource ID'}
                     required={true}
                     disabled={customSettings || disabled}
                     readonly={readonly}
                     readonlyDots={readonly}
                     help={value.id ? `${dynamicResourcePrefix}.${value.id}` : undefined}
                  />
               </div>

               {customSettings ? (
                  <div className={classes.customSettings}>
                     You should use <ExternalLink href={EXTERNAL_LINKS.deployDocs.dctl}>DCTL</ExternalLink> to edit
                     custom settings
                  </div>
               ) : (
                  <>
                     {value.urls?.map((url, i) => (
                        <FieldLayout2
                           // eslint-disable-next-line react/no-array-index-key
                           key={i.toString()}
                           name={`${name}.urls[${i}]`}
                           label={'URL'}
                           required={i === 0}
                           disabled={disabled}
                           readonly={readonly}
                           readonlyDots={readonly}
                           hideErrors={true}
                        >
                           <div className={classes.url}>
                              <SandboxUrlField
                                 name={`${name}.urls[${i}]`}
                                 label={null}
                                 disabled={disabled}
                                 readonly={readonly}
                                 controlProps={{ multiline: true }}
                              />

                              {!readonly && !disabled && value.urls.length > 1 ? (
                                 <RemoveButton onClick={() => handleUrlRemove(i)} />
                              ) : null}
                           </div>
                        </FieldLayout2>
                     ))}

                     <InputField2
                        name={`${name}.storageDir`}
                        label={'Storage dir'}
                        required={true}
                        disabled={disabled}
                        readonly={readonly}
                        readonlyDots={readonly}
                     />

                     <InputField2
                        name={`${name}.destination`}
                        label={'Destination'}
                        required={true}
                        disabled={disabled}
                        readonly={readonly}
                        readonlyDots={readonly}
                     />

                     <div className={readonly ? undefined : classes.formRow}>
                        <InputField2
                           name={`${name}.updateWindow`}
                           label={'Update window'}
                           required={true}
                           disabled={disabled}
                           readonly={readonly}
                           readonlyDots={readonly}
                           controlProps={{ controlProps: { min: 0 }, type: 'number' }}
                        />

                        <InputField2
                           name={`${name}.cachedRevisionsCount`}
                           label={'Cached revisions count'}
                           disabled={disabled}
                           required={true}
                           readonly={readonly}
                           readonlyDots={readonly}
                           controlProps={{ controlProps: { min: 0 }, type: 'number' }}
                        />
                     </div>

                     <NotifyPolicySubForm name={`${name}.notifyPolicy`} disabled={disabled} readonly={readonly} />

                     <AdvancedSettingsSubForm
                        name={`${name}.advancedSettings`}
                        disabled={disabled}
                        readonly={readonly}
                     />
                  </>
               )}
            </FormCard>
         </div>
      );
   },
);

DynamicResourceRecord.displayName = 'DynamicResourceRecord';
