import { action } from '@storybook/addon-actions';
import { boolean } from '@storybook/addon-knobs';
import { Meta, Story } from '@storybook/react';
import { Formik } from 'formik';
import React from 'react';

import { BoxDynamicResource, DynamicResourceNotifyPolicyMode } from '../../../../../models/ui';
import { DevForm } from '../../../../lib';

import { DynamicResourcesSubForm } from './DynamicResourcesSubForm';

const dynamicResources = [
   {
      id: 'httpAction',
      initialId: 'httpAction',
      urls: ['sbr:1234567890'],
      destination: '/destination/http',
      storageDir: '/storage/http',
      updateWindow: 2,
      cachedRevisionsCount: 3,

      notifyPolicy: {
         mode: DynamicResourceNotifyPolicyMode.Http,
         httpAction: {
            url: 'http://url...',
            expectedAnswer: 'http answer',
         },
      },

      advancedSettings: {
         allowDeduplication: false,
         maxDownloadSpeed: null,
         verification: {
            checkPeriodMs: null,
            checksum: null,
         },
      },

      customSettings: {
         deployGroups: false,
         requiredLabels: false,
      },
   },
   {
      id: 'execAction',
      initialId: 'execAction',
      urls: ['sbr:1234567890'],
      destination: '/destination/exec',
      storageDir: '/storage/exec',
      updateWindow: 2,
      cachedRevisionsCount: 3,

      notifyPolicy: {
         mode: DynamicResourceNotifyPolicyMode.Exec,
         execAction: {
            commandLine: 'exec command',
            expectedAnswer: 'exec answer',
         },
      },

      advancedSettings: {
         allowDeduplication: false,
         maxDownloadSpeed: null,
         verification: {
            checkPeriodMs: null,
            checksum: null,
         },
      },

      customSettings: {
         deployGroups: false,
         requiredLabels: false,
      },
   },
   {
      id: 'disabledAction',
      initialId: 'disabledAction',
      urls: ['sbr:1234567890', 'rbtorrent:abc1234567890...'],
      destination: '/destination/disabled',
      storageDir: '/storage/disabled',
      updateWindow: 2,
      cachedRevisionsCount: 15,

      notifyPolicy: {
         mode: DynamicResourceNotifyPolicyMode.Disabled,
      },

      advancedSettings: {
         allowDeduplication: true,
         maxDownloadSpeed: 50,
         verification: {
            checkPeriodMs: 1000,
            checksum: 'checksum',
         },
      },

      customSettings: {
         deployGroups: false,
         requiredLabels: false,
      },
   },
   {
      id: 'empty',
      initialId: 'empty',
      urls: [null],
      destination: null,
      storageDir: null,
      updateWindow: null,
      cachedRevisionsCount: null,

      notifyPolicy: {
         mode: DynamicResourceNotifyPolicyMode.Disabled,
      },

      advancedSettings: {
         allowDeduplication: false,
         maxDownloadSpeed: null,
         verification: {
            checkPeriodMs: null,
            checksum: null,
         },
      },

      customSettings: {
         deployGroups: false,
         requiredLabels: false,
      },
   },
] as BoxDynamicResource[];

const customSettings = [
   {
      id: 'customDeployGroups',
      initialId: 'customDeployGroups',
      urls: ['sbr:1234567890'],
      destination: '/destination/customDeployGroups',
      storageDir: '/storage/customDeployGroups',
      updateWindow: 1,
      cachedRevisionsCount: 5,

      notifyPolicy: {
         mode: DynamicResourceNotifyPolicyMode.Disabled,
      },

      advancedSettings: {
         allowDeduplication: false,
         maxDownloadSpeed: null,
         verification: {
            checkPeriodMs: null,
            checksum: null,
         },
      },

      customSettings: {
         deployGroups: true,
         requiredLabels: false,
      },
   },
   {
      id: 'customRequiredLabels',
      initialId: 'customRequiredLabels',
      urls: ['sbr:1234567890'],
      destination: '/destination/customRequiredLabels',
      storageDir: '/storage/customRequiredLabels',
      updateWindow: 2,
      cachedRevisionsCount: 10,

      notifyPolicy: {
         mode: DynamicResourceNotifyPolicyMode.Disabled,
      },

      advancedSettings: {
         allowDeduplication: false,
         maxDownloadSpeed: null,
         verification: {
            checkPeriodMs: null,
            checksum: null,
         },
      },

      customSettings: {
         deployGroups: false,
         requiredLabels: true,
      },
   },
];

const initialValues = {
   default: dynamicResources,
   customSettings,
   empty: [],
};

export interface Args {
   name?: string;
   readonly?: boolean;
   disabled?: boolean;
}

const Template: Story<Args> = args => (
   <Formik initialValues={initialValues} onSubmit={action('onSubmit')}>
      {() => (
         <>
            <DynamicResourcesSubForm
               name={'default'}
               dynamicResourcePrefix={'stageId.deployUnitId'}
               disabled={args.disabled ?? boolean('disabled', false)}
               readonly={args.readonly ?? boolean('readonly', false)}
               {...args}
            />
            <DevForm />
         </>
      )}
   </Formik>
);

export const Regular = Template.bind({});
Regular.args = {
   disabled: false,
   readonly: false,
};

export const RegularReadonly = Template.bind({});
RegularReadonly.args = {
   disabled: false,
   readonly: true,
};

export const CustomSettings = Template.bind({});
CustomSettings.args = {
   name: 'customSettings',
   disabled: false,
   readonly: false,
};

export const CustomSettingsReadonly = Template.bind({});
CustomSettingsReadonly.args = {
   name: 'customSettings',
   disabled: false,
   readonly: true,
};

export const Empty = Template.bind({});
Empty.args = {
   name: 'empty',
   disabled: false,
   readonly: false,
};

export const EmptyReadonly = Template.bind({});
EmptyReadonly.args = {
   name: 'empty',
   disabled: false,
   readonly: true,
};

export default {
   title: 'components/stage-levels/box/fields/DynamicResourcesSubForm',
} as Meta;
