import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { EMPTY_VALUE, ExternalLink, FormButton, isEmpty, WarningPlate } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback, useMemo } from 'react';

import { EXTERNAL_LINKS } from '../../../../../models';
import {
   Box,
   BoxDynamicResource,
   BoxFormParams,
   DynamicResourceNotifyPolicyMode,
   SidecarName,
   Workload,
} from '../../../../../models/ui';
import { getInitialIds } from '../../../../../models/ui/stage/form-utils';
import { SubForm } from '../../../../huge-form';
import { SidecarResourcePlate } from '../../../../stage-huge-form/components';
import { DynamicResourceRecord } from './DynamicResourceRecord';

import classes from './DynamicResourcesSubForm.module.css';

interface Props {
   disabled: boolean;
   dynamicResourcePrefix: string;
   name: string;
   readonly: boolean;
   form?: SubForm<BoxFormParams, Box, Workload>;
}

export const DynamicResourcesSubForm: React.FC<Props> = React.memo(
   ({ readonly, disabled, dynamicResourcePrefix, name, form: subForm }) => {
      const form = useFormikContext();
      const { value } = form.getFieldMeta<BoxDynamicResource[]>(name);

      const { deployUnit = '', box = '' } = subForm
         ? getInitialIds(subForm.formParams, subForm.parentForms, subForm.levelConfig.id)
         : {};

      const handleAddResource = useCallback(() => {
         const newResource = {
            id: '',
            urls: [null],
            storageDir: null,
            destination: null,
            updateWindow: 1,
            cachedRevisionsCount: 2,
            customSettings: {
               deployGroups: false,
               requiredLabels: false,
            },
            notifyPolicy: {
               mode: DynamicResourceNotifyPolicyMode.Disabled,
            },
            advancedSettings: {
               allowDeduplication: false,
               maxDownloadSpeed: null,
               verification: {
                  checkPeriodMs: null,
                  checksum: null,
               },
            },
         };

         form.setFieldValue(name, [...value, newResource]);
      }, [form, name, value]);

      const handleRemoveResource = useCallback(
         (i: number) => {
            const newValue = [...value];
            newValue.splice(i, 1);

            form.setFieldTouched(`${name}[${i}]`, false);
            form.setFieldValue(name, newValue);
         },
         [form, name, value],
      );

      // eslint-disable-next-line arrow-body-style
      const customSettings = useMemo(() => {
         return {
            deployGroups: value.filter(v => v.customSettings?.deployGroups)?.map(v => v.id) ?? [],
            requiredLabels: value.filter(v => v.customSettings?.requiredLabels)?.map(v => v.id) ?? [],
         };
      }, [value]);

      return (
         <>
            {!isEmpty(customSettings.deployGroups) || !isEmpty(customSettings.requiredLabels) ? (
               <WarningPlate className={classes.warning}>
                  {!isEmpty(customSettings.deployGroups) ? (
                     <p>
                        There are more than one deploy groups in some of your dynamic resources (
                        {customSettings.deployGroups.join(', ')}
                        ). Currently and unfortunately Deploy UI is unable to edit these settings.
                     </p>
                  ) : null}

                  {!isEmpty(customSettings.requiredLabels) ? (
                     <p>
                        Custom required_labels for pods filtering are present in settings of some of your dynamic
                        resources ({customSettings.deployGroups.join(', ')}
                        ). We don't support this feature in Deploy UI yet.
                     </p>
                  ) : null}

                  <p>
                     These resources will be read-only. You can use{' '}
                     <ExternalLink href={EXTERNAL_LINKS.deployDocs.dctl}>DCTL</ExternalLink> to edit any custom setting
                     of your dynamic resources.
                  </p>
               </WarningPlate>
            ) : null}

            {value.length > 0 && (
               <div className={classes.warning}>
                  <SidecarResourcePlate sidecar={SidecarName.DynamicResource} deployUnit={deployUnit} box={box} />
               </div>
            )}

            <div data-test={'Box:DynamicResources'}>
               {value.length === 0 ? (
                  <>{readonly ? <div className={classes.empty}>{EMPTY_VALUE}</div> : null}</>
               ) : (
                  value.map((layer, i) => (
                     <DynamicResourceRecord
                        // eslint-disable-next-line react/no-array-index-key
                        key={i}
                        name={`${name}.[${i}]`}
                        index={i}
                        disabled={disabled}
                        readonly={readonly}
                        onRemove={handleRemoveResource}
                        dynamicResourcePrefix={dynamicResourcePrefix}
                     />
                  ))
               )}

               {disabled || readonly ? null : (
                  <FormButton
                     icon={faPlus}
                     onClick={handleAddResource}
                     skipLeftSpace={true}
                     dataE2e={'DynamicResources:AddResource'}
                  >
                     Add dynamic resource
                  </FormButton>
               )}
            </div>
         </>
      );
   },
);

DynamicResourcesSubForm.displayName = 'DynamicResourcesSubForm';
