import {
   BytesField,
   classNames,
   EnumField2,
   EnumOption,
   ExternalLink,
   FieldLayout2,
   FormAdvancedSection,
   FormErrors,
   InputField2,
   WarningPlate,
} from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback, useContext, useMemo } from 'react';
import { useSelector } from 'react-redux';

import {
   Box,
   DeployUnit,
   DeployUnitFormParams,
   DeployUnitType,
   DeployUnitTypeOptions,
   duPaths,
   isDuAdvancedRequirementsEmpty,
   LayerSourceFileStoragePolicy,
   SidecarName,
   SidecarsForUpdating,
   sidecarsUpdateConfig,
} from '../../../../../models/ui';
import { SecretsReloadContext } from '../../../../../models/ui/secrets';
import { SecretList, SecretsContext } from '../../../../../modules/secrets';
import { ETransmitSystemLogs } from '../../../../../proto-typings';
import type { RootState } from '../../../../../redux';
import { formName } from '../../../../../utils';
import {
   BandwidthField,
   CpuLimitField,
   EnabledSwitcherField,
   FormTab,
   FormTabsWithUrl,
   selectCurrentFormState,
} from '../../../../forms';
import { HugeFormContext, SubFormProps } from '../../../../huge-form';
import { useSecretsContextValueFromForm } from '../../../_common/useSecretsContextValueFromForm';
import { editOnlyForNewProps } from '../../../_common/utils';
import { AntiaffinityRecordField } from '../../fields/AntiaffinityRecordField/AntiaffinityRecordField';
import { RuntimeVersionSubForm } from '../../fields/RuntimeVersionSubForm/RuntimeVersionSubForm';
import {
   AnonymousMemoryLimitHint,
   AntiaffinityHint,
   DeployUnitLocationHint,
   DeployUnitTypeHint,
   DisruptionBudgetHint,
   EndpointSetsHint,
   LayerSourceFileStoragePolicyHint,
   LogbrokerHint,
   MaxTolerableDowntimePodsHint,
   MaxTolerableDowntimeSecondsHint,
   PodAgentHint,
   PodSettingsHint,
   ResourceRequestsHint,
   TransmitSystemLogsHint,
   YasmHint,
} from '../../hints';
import { DeployUnitLocationWarnings } from '../DeployUnitLocationWarnings';
import { DeployUnitSettingsForm } from '../DeployUnitSettingsForm/DeployUnitSettingsForm';
import { DisksSubForm } from '../DisksSubForm/DisksSubForm';
import { EndpointSetsSubForm } from '../EndpointSetsSubForm/EndpointSetsSubForm';
import { LocationsSubForm } from '../LocationsSubForm/LocationsSubForm';
import { LogbrokerConfigSubForm } from '../LogbrokerConfigSubForm/LogbrokerConfigSubForm';
import { NetworkSubForm } from '../NetworkSubForm/NetworkSubForm';
import { NodeFiltersSubForm } from '../NodeFiltersSubForm/NodeFiltersSubForm';
import { PerLocationSettingsSubForm } from '../PerLocationSettingsSubForm/PerLocationSettingsSubForm';
import { SidecarSubForm } from '../SidecarSubForm/SidecarSubForm';
import { TvmSubForm } from '../TvmSubForm/TvmSubForm';
import { YasmSubForm } from '../YasmSubForm/YasmSubForm';

import classes from './DeployUnitForm.module.css';

const rowFieldNames = ['disruptionBudget', 'maxTolerableDowntimePods', 'maxTolerableDowntimeSeconds'];

const defaultLayerSourceFileStoragePolicyOptions: EnumOption[] = [
   {
      value: LayerSourceFileStoragePolicy.None,
      title: 'None',
   },
   {
      value: LayerSourceFileStoragePolicy.Keep,
      title: 'Keep',
   },
   {
      value: LayerSourceFileStoragePolicy.Remove,
      title: 'Remove',
   },
];

const transmitSystemLogsOptions: EnumOption[] = [
   {
      value: ETransmitSystemLogs.ETransmitSystemLogsPolicy_NONE,
      title: 'None',
   },
   {
      value: ETransmitSystemLogs.ETransmitSystemLogsPolicy_DISABLED,
      title: 'Disabled',
   },
   {
      value: ETransmitSystemLogs.ETransmitSystemLogsPolicy_ENABLED,
      title: 'Enabled',
   },
];

const sidecarsHints: Partial<Record<SidecarName, ReactNode>> = {
   [SidecarName.Logbroker]: <LogbrokerHint />,
   [SidecarName.PodAgent]: <PodAgentHint />,
};

export const DeployUnitForm: React.FC<SubFormProps<DeployUnitFormParams, DeployUnit, Box>> = ({
   disabled,
   form,
   formik,
   isRootEntityNew,
   readonly,
}) => {
   const { payload } = useContext(HugeFormContext);
   const reloadSecretUsages = useContext(SecretsReloadContext);

   const { id: stageFormId, formParams: stageFormParams } = form.parentForms[0];
   const parentFormValuesSelector = useCallback((s: RootState) => selectCurrentFormState(s, stageFormId)?.values, [
      stageFormId,
   ]);

   const parentFormValues = useSelector(parentFormValuesSelector);
   const actualStageId = parentFormValues?.id ?? stageFormParams.id;
   const actualSoxService = parentFormValues?.soxService ?? stageFormParams.soxService;

   const enabledLocations = useMemo(
      () => Object.keys(formik.values.locations).filter(location => formik.values.locations[location].enabled),
      [formik.values.locations],
   );

   const formElement = (
      <div className={classes.form}>
         <InputField2
            name={formName(duPaths.id)}
            label={'Deploy Unit ID'}
            help={isRootEntityNew ? 'Deploy unit ID can not be changed after creation' : undefined}
            {...editOnlyForNewProps(readonly, disabled, form.isAdded, isRootEntityNew)}
            readonlyDots={true}
         />

         {/* TODO: стоит уточнить, это ещё актуально? */}
         {payload.showDiskIsolationField || !form.formParams.tempDiskIsolation ? (
            <EnabledSwitcherField
               name={formName(duPaths.tempDiskIsolation)}
               label={'Disk isolation'}
               disabled={disabled}
               readonly={readonly}
               help={
                  <WarningPlate className={classes.warning}>
                     Beware that enabling and disabling disk isolation will cause mass pods reallocation.{' '}
                     <ExternalLink href={'https://clubs.at.yandex-team.ru/infra-cloud/1074'}>Read more</ExternalLink>
                  </WarningPlate>
               }
               readonlyDots={true}
            />
         ) : null}

         <RuntimeVersionSubForm
            name={formName(duPaths.patchersRevision.value)}
            disabled={disabled}
            readonly={readonly}
         />

         <NetworkSubForm name={formName(duPaths.networkDefaults)} readonly={readonly} disabled={disabled} />

         <FieldLayout2
            name={formName(duPaths.locations)}
            label={'Location policy'}
            hideErrors={true}
            readonly={readonly}
            disabled={disabled}
            hint={<DeployUnitLocationHint />}
            bigLabel={true}
         >
            <EnumField2
               name={formName(duPaths.type)}
               label={'Deploy unit type'}
               controlProps={{ options: DeployUnitTypeOptions }}
               help={isRootEntityNew ? 'Deploy unit type can not be changed after creation' : undefined}
               hint={<DeployUnitTypeHint />}
               {...editOnlyForNewProps(readonly, disabled, form.isAdded, isRootEntityNew)}
               readonlyDots={true}
            />

            <div className={classNames({ [classes.row]: !readonly })}>
               <AntiaffinityRecordField
                  name={formName(duPaths.antiaffinity)}
                  label={'Antiaffinity'}
                  hidden={formik.values.type === DeployUnitType.MultiCluster}
                  hint={<AntiaffinityHint />}
                  disabled={disabled}
                  readonly={readonly}
                  readonlyDots={true}
               />

               <InputField2
                  name={formName(duPaths.disruptionBudget)}
                  label={'Disruption budget'}
                  hidden={formik.values.type === DeployUnitType.PerCluster}
                  disabled={disabled}
                  readonly={readonly}
                  required={formik.values.type === DeployUnitType.MultiCluster}
                  controlProps={{ controlProps: { min: 0 }, type: 'number' }}
                  hint={<DisruptionBudgetHint />}
                  hideErrors={true}
                  readonlyDots={true}
               />

               <InputField2
                  name={formName(duPaths.maxTolerableDowntimePods)}
                  label={'Maintenance budget'}
                  hidden={formik.values.type === DeployUnitType.PerCluster}
                  disabled={disabled}
                  readonly={readonly}
                  controlProps={{ controlProps: { min: 0 }, type: 'number' }}
                  hint={<MaxTolerableDowntimePodsHint />}
                  hideErrors={true}
                  readonlyDots={true}
               />

               <InputField2
                  name={formName(duPaths.maxTolerableDowntimeSeconds)}
                  label={'Max maintenance duration'}
                  hidden={formik.values.type === DeployUnitType.PerCluster}
                  disabled={disabled}
                  readonly={readonly}
                  controlProps={{ controlProps: { min: 0 }, type: 'number' }}
                  hint={<MaxTolerableDowntimeSecondsHint />}
                  hideErrors={true}
                  readonlyDots={true}
               />
            </div>

            <FormErrors names={rowFieldNames} />

            <DeployUnitLocationWarnings />

            <LocationsSubForm
               name={formName(duPaths.locations)}
               readonly={readonly}
               disabled={disabled}
               deployUnitType={formik.values.type}
            />

            <PerLocationSettingsSubForm
               locations={enabledLocations}
               name={formName(duPaths.perLocationSettings)}
               readonly={readonly}
               disabled={disabled}
               deployUnitType={formik.values.type}
            />
         </FieldLayout2>

         <FieldLayout2
            name={'podSettings'}
            label={'Per pod settings'}
            readonly={readonly}
            disabled={disabled}
            hint={<PodSettingsHint />}
            hideErrors={true}
            bigLabel={true}
         >
            <CpuLimitField
               name={formName(duPaths.cpu)}
               label={'CPU'}
               disabled={disabled}
               readonly={readonly}
               required={true}
               hint={<ResourceRequestsHint />}
               readonlyDots={readonly}
            />

            <BytesField
               name={formName(duPaths.ram)}
               label={'RAM'}
               disabled={disabled}
               readonly={readonly}
               required={true}
               hint={<ResourceRequestsHint />}
               readonlyDots={readonly}
            />

            <BytesField
               name={formName(duPaths.anonymousMemoryLimit)}
               label={'Anonymous memory limit'}
               disabled={disabled}
               readonly={readonly}
               hint={<AnonymousMemoryLimitHint />}
               readonlyDots={readonly}
            />

            <BandwidthField
               name={formName(duPaths.networkBandwidth.guarantee)}
               label={'Network bandwidth guarantee'}
               disabled={disabled}
               readonly={readonly}
               hint={<ResourceRequestsHint />}
               readonlyDots={readonly}
            />

            <BandwidthField
               name={formName(duPaths.networkBandwidth.limit)}
               label={'Network bandwidth limit'}
               disabled={disabled}
               readonly={readonly}
               hint={<ResourceRequestsHint />}
               readonlyDots={readonly}
            />

            <FormAdvancedSection
               header={null}
               toggleText={'Advanced requirements'}
               wrapperClassName={classes.formSection}
               openedByDefault={!isDuAdvancedRequirementsEmpty(form.value)}
            >
               <NodeFiltersSubForm disabled={disabled} name={'nodeFilters'} readonly={readonly} />
            </FormAdvancedSection>
         </FieldLayout2>

         <DeployUnitSettingsForm name={formName(duPaths.settings)} readonly={readonly} disabled={disabled} />

         <FieldLayout2
            name={'endpointSets'}
            label={'Endpoint sets'}
            hint={<EndpointSetsHint />}
            hideErrors={true}
            bigLabel={true}
         >
            <EndpointSetsSubForm
               name={formName(duPaths.endpointSets)}
               readonly={readonly}
               disabled={disabled}
               endpointSetPrefix={`${actualStageId}.${formik.values.id}`}
            />
         </FieldLayout2>

         <TvmSubForm name={formName(duPaths.tvm)} readonly={readonly} disabled={disabled} form={form} />

         <FieldLayout2 name={'yasm'} label={'Monitoring in YASM'} hint={<YasmHint />} bigLabel={true}>
            <YasmSubForm name={formName(duPaths.yasm)} readonly={readonly} disabled={disabled} />
         </FieldLayout2>

         <EnabledSwitcherField
            name={formName(duPaths.collectPortometricsFromSidecars)}
            label={'Enable portometrics'}
            disabled={disabled}
            readonly={readonly}
            bigLabel={true}
         />

         {/* update sidecars */}

         <LogbrokerConfigSubForm name={formName(duPaths.logbrokerConfig)} disabled={disabled} readonly={readonly} />

         <EnumField2
            name={formName(duPaths.transmitSystemLogs)}
            label={'Transmit system logs policy'}
            readonly={readonly}
            disabled={disabled}
            controlProps={{ options: transmitSystemLogsOptions }}
            bigLabel={true}
            hint={<TransmitSystemLogsHint />}
         />

         <EnabledSwitcherField
            name={formName(duPaths.infraComponents.allowAutomaticUpdates)}
            disabled={disabled}
            readonly={readonly}
            label={'Runtime/sidecars autoupdate'}
            bigLabel={true}
         />

         {SidecarsForUpdating.map(sidecarName => (
            <FieldLayout2
               key={sidecarName}
               name={sidecarName}
               label={sidecarsUpdateConfig[sidecarName].title}
               hideErrors={true}
               hint={sidecarsHints[sidecarName]}
               bigLabel={true}
            >
               <SidecarSubForm
                  name={formName(duPaths.sidecars[sidecarName])}
                  readonly={readonly}
                  disabled={disabled}
                  sidecarName={sidecarsUpdateConfig[sidecarName].title}
                  sidecarSandboxId={sidecarsUpdateConfig[sidecarName].sidecarSandboxId}
               />
            </FieldLayout2>
         ))}
      </div>
   );

   return (
      <SecretsContext.Provider value={useSecretsContextValueFromForm(form, reloadSecretUsages)}>
         <FormTabsWithUrl defaultTab={'form'}>
            <FormTab id={'form'} title={'Form'}>
               {formElement}
            </FormTab>
            <FormTab
               id={'disks'}
               title={'Disks, volumes and resources'}
               promo={{
                  key: 'promo.visible.du.tabs.disks.resources',
                  message: 'Disks, volumes, layers and static resources are here now!',
               }}
            >
               <div className={classes.form}>
                  <div
                     className={classes.defaultLayerSourceFileStoragePolicy}
                     data-test={'DefaultLayerSourceFileStoragePolicy'}
                  >
                     <EnumField2
                        name={formName(duPaths.defaultLayerSourceFileStoragePolicy)}
                        label={'Default layer storage policy'}
                        readonly={readonly}
                        disabled={disabled}
                        controlProps={{ options: defaultLayerSourceFileStoragePolicyOptions }}
                        bigLabel={true}
                        hint={<LayerSourceFileStoragePolicyHint />}
                     />
                  </div>

                  <DisksSubForm
                     name={formName(duPaths.disks)}
                     readonly={readonly}
                     disabled={disabled}
                     soxService={actualSoxService}
                  />
               </div>
            </FormTab>
            <FormTab id={'secrets'} title={'Secrets'}>
               <SecretList disabled={disabled} readonly={readonly} />
            </FormTab>
         </FormTabsWithUrl>
      </SecretsContext.Provider>
   );
};

DeployUnitForm.displayName = 'DeployUnitForm';
