import { Suggest } from '@yandex-data-ui/common';
import { classNames, EMPTY_VALUE, FieldLayout2, hasIncluded, useDismounted } from '@yandex-infracloud-ui/libs';
import { getIn, useFormikContext } from 'formik';
import React, { useCallback, useEffect, useState } from 'react';
import { map, takeUntil } from 'rxjs/operators';
import { solomonApi } from '../../../../../../services';
import classes from './NotificationChannelSubForm.module.css';

interface Props {
   name: string;
   label: string;
   readonly: boolean;
   disabled: boolean;
   project: string;
}

export const NotificationChannelSubForm: React.FC<Props> = React.memo(props => {
   const { name, readonly, disabled, project } = props;
   const form = useFormikContext();
   const value: string = getIn(form.values, name);

   const [query, setQuery] = useState(value ?? '');

   const dismounted = useDismounted();

   useEffect(() => {
      setQuery(value ?? '');
   }, [value]);

   const listChannels = useCallback(
      (q: string) =>
         solomonApi
            .getChannels(project)
            .pipe(
               map(resp => resp.map(item => item.id)),
               map(resp => resp.filter(item => hasIncluded(q, item))),
               takeUntil(dismounted),
            )
            .toPromise()
            .then(x => x ?? []),
      [dismounted, project],
   );

   const handleItemClick = useCallback(
      (item: string) => {
         setQuery(item);
         form.setFieldValue(name, item);
      },
      [form, name],
   );

   const handleQueryChange = useCallback(
      (v: string) => {
         setQuery(v);
         form.setFieldValue(name, v);
      },
      [form, name],
   );

   return (
      <FieldLayout2 {...props} readonlyDots={readonly}>
         {readonly ? (
            <div className={classNames({ [classes.readonly]: readonly })}>{query || EMPTY_VALUE}</div>
         ) : (
            <Suggest<string>
               debounce={150}
               disabled={disabled}
               getItems={listChannels}
               getItemsOnMount={true}
               itemHeight={35}
               onItemClick={handleItemClick}
               onUpdate={handleQueryChange}
               placeholder={'Type to search...'}
               renderItem={item => <div className={classes.item}>{item}</div>}
               showItemsOnNoText={true}
               text={query}
               virtualized={true}
            />
         )}
      </FieldLayout2>
   );
});

NotificationChannelSubForm.displayName = 'NotificationChannelSubForm';
