import {
   BytesField,
   CheckboxField2,
   EnumField2,
   FieldLayout2,
   InputField2,
   WarningPlate,
} from '@yandex-infracloud-ui/libs';
import { getIn, useFormikContext } from 'formik';
import React, { useEffect, useMemo } from 'react';

import { HDD_BANDWIDTH_LIMIT_FACTOR, SSD_BANDWIDTH_LIMIT_FACTOR } from '../../../../../models/constants';

import { DeployUnitDisk, DiskType, DiskTypeOptions } from '../../../../../models/ui';
import { BandwidthField } from '../../../../forms';
import { DiskLayersSubForm } from '../../fields/DiskLayersSubForm/DiskLayersSubForm';
import { DiskStaticResourcesSubForm } from '../../fields/DiskStaticResourcesSubForm/DiskStaticResourcesSubForm';
import { DiskVolumesSubForm } from '../../fields/DiskVolumesSubForm/DiskVolumesSubForm';

import classes from './DiskSubForm.module.css';

import { DiskBandwidthGuaranteeHint, DiskBandwidthLimitHint, DiskVolumeRequestsHint } from './hints';

interface Props {
   disabled: boolean;
   index: number;
   name: string;
   readonly: boolean;

   soxService: boolean;
}

export const DiskSubForm: React.FC<Props> = React.memo(({ readonly, disabled, index, name, soxService }) => {
   const form = useFormikContext();
   const value: DeployUnitDisk = getIn(form.values, name);

   const defaultSettings = useMemo(() => value.bandwidth.limit.defaultSettings ?? false, [value]);

   const defaultLimit = useMemo(() => value.bandwidth.limit.default ?? null, [value]);

   const customLimit = useMemo(() => value.bandwidth.limit.custom ?? null, [value]);

   const defaultBandwidthLimit = useMemo(() => {
      const guarantee = value.bandwidth.guarantee ?? null;

      if (!guarantee) {
         return null;
      }

      return value.type === DiskType.SSD
         ? guarantee * SSD_BANDWIDTH_LIMIT_FACTOR
         : guarantee * HDD_BANDWIDTH_LIMIT_FACTOR;
   }, [value]);

   useEffect(() => {
      if (defaultSettings && defaultLimit !== defaultBandwidthLimit) {
         form.setFieldValue(`${name}.bandwidth.limit.default`, defaultBandwidthLimit);
      }
   }, [form, name, defaultSettings, defaultLimit, defaultBandwidthLimit]);

   return (
      <>
         <FieldLayout2
            name={name}
            label={`Disk ${index + 1}`}
            hint={<DiskVolumeRequestsHint />}
            bigLabel={true}
            hideErrors={true}
         >
            <InputField2
               name={`${name}.id`}
               label={'ID'}
               required={true}
               disabled={true}
               readonly={readonly}
               readonlyDots={readonly}
            />

            <EnumField2
               name={`${name}.type`}
               label={'Type'}
               controlProps={{ options: DiskTypeOptions }}
               required={true}
               disabled={disabled}
               readonly={readonly}
               readonlyDots={readonly}
            />

            <BytesField
               name={`${name}.size`}
               label={'Size'}
               required={true}
               disabled={disabled}
               readonly={readonly}
               readonlyDots={readonly}
            />

            <BandwidthField
               name={`${name}.bandwidth.guarantee`}
               label={'Bandwidth guarantee'}
               controlProps={{ useRenderDetails: true }}
               required={true}
               disabled={disabled}
               readonly={readonly}
               hint={<DiskBandwidthGuaranteeHint />}
               readonlyDots={readonly}
            />

            <div className={classes.bandwidthLimit}>
               <BandwidthField
                  name={defaultSettings ? `${name}.bandwidth.limit.default` : `${name}.bandwidth.limit.custom`}
                  label={'Bandwidth limit'}
                  controlProps={{ useRenderDetails: true }}
                  required={value.bandwidth.guarantee !== null}
                  disabled={disabled || defaultSettings}
                  readonly={readonly}
                  hint={<DiskBandwidthLimitHint />}
                  readonlyDots={readonly}
               />
            </div>

            {!readonly ? (
               <CheckboxField2
                  name={`${name}.bandwidth.limit.defaultSettings`}
                  disabled={disabled}
                  readonly={readonly}
                  label={null}
                  controlProps={{ disableIndeterminate: true }}
               >
                  Use default bandwidth limit
               </CheckboxField2>
            ) : null}

            {!readonly && !defaultSettings && customLimit !== defaultBandwidthLimit ? (
               <WarningPlate>
                  If you are not sure what you are doing, it would be better to use default bandwidth limit
               </WarningPlate>
            ) : null}

            <div className={classes.staticResources}>
               <DiskStaticResourcesSubForm
                  name={`${name}.staticResources`}
                  readonly={readonly}
                  disabled={disabled}
                  soxService={soxService}
               />
            </div>

            <div className={classes.layers}>
               <DiskLayersSubForm name={`${name}.layers`} readonly={readonly} disabled={disabled} />
            </div>

            <div className={classes.volumes}>
               <DiskVolumesSubForm
                  name={`${name}.volumes`}
                  readonly={readonly}
                  disabled={disabled}
                  layers={value.layers}
                  staticResources={value.staticResources}
               />
            </div>
         </FieldLayout2>
      </>
   );
});

DiskSubForm.displayName = 'DiskSubForm';
