import { faUndo } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';
import { FormCard, InputField2, RemoveButton } from '@yandex-infracloud-ui/libs';
import { getIn, useFormikContext } from 'formik';
import React, { useCallback } from 'react';

import { DeployUnitEndpointSet } from '../../../../../models/ui';

import classes from './EndpointSetRecord.module.css';

interface Props {
   disabled: boolean;
   endpointSetPrefix: string;
   index: number;
   name: string;
   readonly: boolean;

   onRemove(i: number): void;

   onRestore(i: number): void;
}

export const EndpointSetRecord = React.memo(
   ({ index, name, readonly, disabled, endpointSetPrefix, onRemove, onRestore }: Props) => {
      const form = useFormikContext();
      const value: DeployUnitEndpointSet = getIn(form.values, name);

      const handleRemoveEndpointSet = useCallback(() => onRemove!(index!), [index, onRemove]);
      const handleRestoreEndpointSet = useCallback(() => onRestore!(index!), [index, onRestore]);

      const endpointSetName = value.id ? `${endpointSetPrefix}.${value.id}` : endpointSetPrefix;

      return (
         <FormCard
            title={endpointSetName}
            className={value.removed ? classes.removed : undefined}
            buttons={
               !readonly && !disabled ? (
                  <>
                     {value.removed ? (
                        <div data-e2e={'EndpointSet:RestoreEndpointSet'}>
                           <Button view={'flat'} title={'Restore'} onClick={handleRestoreEndpointSet}>
                              <FontAwesomeIcon icon={faUndo} />
                           </Button>
                        </div>
                     ) : (
                        <div data-e2e={'EndpointSet:RemoveEndpointSet'}>
                           <RemoveButton onClick={handleRemoveEndpointSet} size={'lg'} />
                        </div>
                     )}
                  </>
               ) : null
            }
         >
            {value.removed === true ? (
               'removed'
            ) : (
               <>
                  <div className={classes.id}>
                     <InputField2
                        name={`${name}.id`}
                        label={'Endpoint set ID'}
                        /* eslint-disable-next-line no-underscore-dangle */
                        disabled={disabled || value._order !== undefined}
                        readonly={readonly}
                        readonlyDots={readonly}
                     />
                  </div>

                  <div className={classes.port}>
                     <InputField2
                        name={`${name}.port`}
                        label={'Port'}
                        controlProps={{ type: 'number', controlProps: { min: 0 } }}
                        /* eslint-disable-next-line no-underscore-dangle */
                        disabled={disabled || value._order !== undefined}
                        readonly={readonly}
                        readonlyDots={readonly}
                     />
                  </div>

                  <div className={classes.liveness_limit_ratio}>
                     <InputField2
                        name={`${name}.liveness_limit_ratio`}
                        label={'Liveness limit ratio'}
                        controlProps={{ type: 'number', controlProps: { min: 0, max: 1, step: 0.05 } }}
                        disabled={disabled} // liveness_limit_ratio можно менять после DEPLOY-4576
                        readonly={readonly}
                        readonlyDots={readonly}
                     />
                  </div>
               </>
            )}
         </FormCard>
      );
   },
);

EndpointSetRecord.displayName = 'EndpointSetRecord';
