import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { FormButton, isEmpty, WarningPlate } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback } from 'react';

import {
   DEFAULT_LIVENESS_LIMIT_RATIO_FOR_DEFAULT_ENDPOINT_SET,
   DEFAULT_PORT_FOR_DEFAULT_ENDPOINT_SET,
} from '../../../../../models';

import { DeployUnitDefaultEndpointSet, DeployUnitEndpointSet } from '../../../../../models/ui';
import { DefaultEndpoint } from './DefaultEndpoint';

import { EndpointSetRecord } from './EndpointSetRecord';

import classes from './EndpointSetsSubForm.module.css';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
   endpointSetPrefix: string;
}

export const EndpointSetsSubForm = React.memo(({ disabled, name, readonly, endpointSetPrefix }: Props) => {
   /* eslint-disable no-underscore-dangle */
   const form = useFormikContext();
   const { value } = form.getFieldMeta<DeployUnitEndpointSet[]>(name);

   const handleAddEndpointSet = useCallback(() => {
      const newValue: DeployUnitEndpointSet[] = [
         ...value,
         {
            id: null,
            port: DEFAULT_PORT_FOR_DEFAULT_ENDPOINT_SET,
            liveness_limit_ratio: DEFAULT_LIVENESS_LIMIT_RATIO_FOR_DEFAULT_ENDPOINT_SET,
         },
      ];

      const last = newValue.length - 1;

      form.setFieldValue(name, newValue);
      form.setFieldTouched(`${name}[${last}].id`, true);
   }, [form, name, value]);

   const handleRemoveEndpointSet = useCallback(
      (i: number) => {
         if (value[i]._order === undefined) {
            const newValue = [...value];

            newValue.splice(i, 1);

            form.setFieldTouched(`${name}[${i}]`, false);
            form.setFieldValue(name, newValue);
         } else {
            const newValue: DeployUnitEndpointSet[] = [...value].map((v, index) =>
               index === i
                  ? {
                       ...v,
                       removed: true,
                    }
                  : v,
            );

            form.setFieldValue(name, newValue);
         }
      },
      [form, name, value],
   );

   const handleRestoreEndpointSet = useCallback(
      (i: number) => {
         const newValue: DeployUnitEndpointSet[] = [...value].map((v, index) =>
            index === i
               ? {
                    ...v,
                    removed: false,
                 }
               : v,
         );

         form.setFieldValue(name, newValue);
      },
      [form, name, value],
   );

   const showDefaultEndpointSet = value.every(set => set.id) || isEmpty(value);

   return (
      <>
         {!isEmpty(value)
            ? value.map((endpointSet: DeployUnitEndpointSet, i: number) => (
                 <EndpointSetRecord
                    // eslint-disable-next-line react/no-array-index-key
                    key={i}
                    index={i}
                    name={`${name}[${i}]`}
                    disabled={disabled}
                    readonly={readonly}
                    onRemove={handleRemoveEndpointSet}
                    onRestore={handleRestoreEndpointSet}
                    endpointSetPrefix={endpointSetPrefix}
                 />
              ))
            : null}

         {showDefaultEndpointSet && readonly ? (
            <DefaultEndpoint readonly={readonly} endpointSetPrefix={endpointSetPrefix} />
         ) : null}

         {disabled || readonly ? null : (
            <div>
               <FormButton
                  onClick={handleAddEndpointSet}
                  icon={faPlus}
                  dataE2e={'EndpointSets:AddEndpointSetButton'}
                  skipLeftSpace={true}
               >
                  Add endpoint set
               </FormButton>
            </div>
         )}

         {showDefaultEndpointSet && !readonly ? (
            <div className={classes.defaultEndpointBlock}>
               <WarningPlate theme={'info'}>
                  Default endpoint set will be created with params: id <b>{`${endpointSetPrefix}`}</b>, protocol{' '}
                  <b>{DeployUnitDefaultEndpointSet.protocol}</b>, port <b>{DeployUnitDefaultEndpointSet.port}</b>,
                  liveness_limit_ratio <b>{DeployUnitDefaultEndpointSet.liveness_limit_ratio}</b>
               </WarningPlate>
            </div>
         ) : null}
      </>
   );
   /* eslint-enable no-underscore-dangle */
});

EndpointSetsSubForm.displayName = 'EndpointSetsSubForm';
