import { CheckboxField2, classNames, FormCard, FormErrors, Hint, InputField2 } from '@yandex-infracloud-ui/libs';
import { useField } from 'formik';
import React, { useMemo } from 'react';

import { DeployUnitLocation, DeployUnitType } from '../../../../../models/ui';
import { AntiaffinityRecordField } from '../../fields/AntiaffinityRecordField/AntiaffinityRecordField';

import {
   AntiaffinityHint,
   DisruptionBudgetHint,
   MaxTolerableDowntimePodsHint,
   MaxTolerableDowntimeSecondsHint,
} from '../../hints';

import classes from './LocationCard.module.css';

interface Props {
   clusterName: string;
   deployUnitType: DeployUnitType;
   disabled: boolean;
   name: string;
   readonly: boolean;
}

export const LocationCard: React.FC<Props> = React.memo(({ clusterName, deployUnitType, disabled, readonly, name }) => {
   const [field] = useField<DeployUnitLocation | undefined>(name);

   const hasValue = field.value?.enabled ?? false;
   const numberControlProps = {
      controlAttrs: { min: 0 },
      type: 'number' as const,
   };

   const firstRowFieldNames = useMemo(
      () => [
         `${name}.podCount`,
         `${name}.disruptionBudget`,
         `${name}.maxTolerableDowntimePods`,
         `${name}.maxTolerableDowntimeSeconds`,
      ],
      [name],
   );

   if (readonly && !field.value?.enabled) {
      return null;
   }

   const header = <h3 className={classes.title}>{clusterName}</h3>;

   return (
      <FormCard
         title={null}
         className={classNames(classes.card, { [classes.enabled]: hasValue, [classes.readonly]: readonly })}
         qa={`LocationCard:${clusterName}`}
      >
         <header className={classes.header}>
            {readonly ? (
               header
            ) : (
               <CheckboxField2
                  name={`${name}.enabled`}
                  label={null}
                  disabled={disabled}
                  readonly={readonly}
                  controlProps={{ disableIndeterminate: true, size: 'l' } as any} // FIXME: fix CheckboxFieldControlProps
                  bigLabel={true}
               >
                  {header}
               </CheckboxField2>
            )}
         </header>

         {hasValue ? (
            <div className={classes.fields}>
               <div className={classes.row}>
                  <InputField2
                     name={`${name}.podCount`}
                     label={'Pods quantity'}
                     controlProps={numberControlProps}
                     required={true}
                     disabled={disabled || !hasValue}
                     readonly={readonly}
                     hideErrors={true}
                     readonlyDots={readonly}
                  />

                  {deployUnitType === DeployUnitType.PerCluster ? (
                     <>
                        <InputField2
                           name={`${name}.disruptionBudget`}
                           label={'Disruption budget'}
                           controlProps={numberControlProps}
                           hint={<DisruptionBudgetHint />}
                           required={true}
                           disabled={disabled || !hasValue}
                           readonly={readonly}
                           hideErrors={true}
                           readonlyDots={readonly}
                        />

                        <InputField2
                           name={`${name}.maxTolerableDowntimePods`}
                           label={'Maintenance budget'}
                           controlProps={numberControlProps}
                           hint={<MaxTolerableDowntimePodsHint />}
                           disabled={disabled || !hasValue}
                           readonly={readonly}
                           hideErrors={true}
                           readonlyDots={readonly}
                        />

                        <InputField2
                           name={`${name}.maxTolerableDowntimeSeconds`}
                           label={'Max maintenance duration'}
                           controlProps={numberControlProps}
                           hint={<MaxTolerableDowntimeSecondsHint />}
                           disabled={disabled || !hasValue}
                           readonly={readonly}
                           hideErrors={true}
                           readonlyDots={readonly}
                        />
                     </>
                  ) : null}
               </div>

               <FormErrors names={firstRowFieldNames} hideEmpty={true} />

               {deployUnitType === DeployUnitType.MultiCluster ? (
                  <>
                     {!readonly && (
                        <div className={classes.antiaffinity}>
                           Antiaffinity <Hint text={<AntiaffinityHint />} />
                        </div>
                     )}
                     <AntiaffinityRecordField
                        name={`${name}.antiaffinity`}
                        disabled={disabled || !hasValue}
                        readonly={readonly}
                        label={readonly ? 'Antiaffinity' : null}
                        readonlyDots={readonly}
                     />
                  </>
               ) : null}
            </div>
         ) : null}
      </FormCard>
   );
});

LocationCard.displayName = 'LocationCard';
