import { DurationField, FieldLayout2, InputField2, isEmpty } from '@yandex-infracloud-ui/libs';
import { getIn, useFormikContext } from 'formik';
import React, { useMemo } from 'react';

import { LogbrokerConfig } from '../../../../../models/ui';

import { SecretSubForm, useSecretsContext } from '../../../../../modules/secrets';
import { EnabledSwitcherField, FormSubSection } from '../../../../forms';

import {
   LogbrokerCustomTopicRequestHint,
   LogbrokerDestroyPolicyHint,
   LogbrokerMaxTriesHint,
   LogbrokerRestartPeriodHint,
   RequiredCustomTopicRequestFieldHint,
   LogbrokerPodAdditionalResourcesHint,
   LogbrokerCustomCpuHint,
} from './hints';

import classes from './LogbrokerConfigSubForm.module.css';

interface Props {
   name: string;
   disabled: boolean;
   readonly: boolean;
}

export const LogbrokerConfigSubForm = React.memo(({ name, disabled, readonly }: Props) => {
   const form = useFormikContext();
   const value: LogbrokerConfig = getIn(form.values, name);

   const { stageId, duId, reloadUsages } = useSecretsContext();

   const isRequiredCustomTopicRequestField = useMemo(
      () => (value?.customTopicRequest?.topicName ? !isEmpty(value.customTopicRequest.topicName) : false),
      [value.customTopicRequest.topicName],
   );

   return (
      <FieldLayout2
         name={name}
         label={'Logbroker config'}
         bigLabel={true}
         disabled={disabled}
         readonly={readonly}
         hideErrors={true}
      >
         <div className={classes.wrapper}>
            <FormSubSection>
               <div className={classes.destroyPolicy}>
                  <FieldLayout2
                     name={`${name}.destroyPolicy`}
                     label={'Destroy policy'}
                     bigLabel={true}
                     disabled={disabled}
                     readonly={readonly}
                     hint={<LogbrokerDestroyPolicyHint />}
                     hideErrors={true}
                  >
                     <div className={!readonly ? classes.row : undefined}>
                        <InputField2
                           name={`${name}.destroyPolicy.maxTries`}
                           label={'Max tries'}
                           disabled={disabled}
                           readonly={readonly}
                           controlProps={{ controlProps: { min: 1 }, type: 'number' }}
                           hint={<LogbrokerMaxTriesHint />}
                           readonlyDots={readonly}
                        />

                        <DurationField
                           name={`${name}.destroyPolicy.restartPeriodMs`}
                           label={'Restart period'}
                           hint={<LogbrokerRestartPeriodHint />}
                           disabled={disabled}
                           readonly={readonly}
                           controlProps={{
                              useMilliseconds: true,
                           }}
                           readonlyDots={readonly}
                        />
                     </div>
                  </FieldLayout2>
               </div>

               <div className={classes.customTopicRequest}>
                  <FieldLayout2
                     name={`${name}.customTopicRequest`}
                     label={'Custom topic request'}
                     bigLabel={true}
                     disabled={disabled}
                     readonly={readonly}
                     hint={<LogbrokerCustomTopicRequestHint />}
                     hideErrors={true}
                  >
                     <div className={!readonly ? classes.row : undefined}>
                        <InputField2
                           name={`${name}.customTopicRequest.topicName`}
                           label={'Topic name'}
                           disabled={disabled}
                           readonly={readonly}
                           readonlyDots={readonly}
                        />

                        <InputField2
                           name={`${name}.customTopicRequest.tvmClientId`}
                           label={'TVM client ID'}
                           disabled={disabled}
                           readonly={readonly}
                           controlProps={{ controlProps: { min: 1 }, type: 'number' }}
                           readonlyDots={readonly}
                           required={isRequiredCustomTopicRequestField}
                           hint={<RequiredCustomTopicRequestFieldHint />}
                        />
                     </div>

                     <FieldLayout2
                        name={`${name}.customTopicRequest.secret`}
                        label={'Secret'}
                        readonly={readonly}
                        readonlyDots={readonly}
                        required={isRequiredCustomTopicRequestField}
                        hint={<RequiredCustomTopicRequestFieldHint />}
                     >
                        <SecretSubForm
                           duId={duId}
                           name={`${name}.customTopicRequest.secret`}
                           stageId={stageId}
                           layout={'vertical'}
                           disabled={false}
                           readonly={readonly}
                           onChange={reloadUsages}
                        />
                     </FieldLayout2>
                  </FieldLayout2>
               </div>

               <div className={classes.podAdditionalResourcesRequest}>
                  <FieldLayout2
                     name={`${name}.podAdditionalResourcesRequest`}
                     label={'Additional resources per pod'}
                     bigLabel={true}
                     disabled={disabled}
                     readonly={readonly}
                     hint={<LogbrokerPodAdditionalResourcesHint />}
                     hideErrors={true}
                  >
                     <EnabledSwitcherField
                        name={`${name}.podAdditionalResourcesRequest.setCpuToZero`}
                        label={'Set CPU to zero'}
                        disabled={disabled}
                        readonly={readonly}
                        hint={<LogbrokerCustomCpuHint />}
                        readonlyDots={true}
                     />
                  </FieldLayout2>
               </div>
            </FormSubSection>
         </div>
      </FieldLayout2>
   );
});

LogbrokerConfigSubForm.displayName = 'LogbrokerConfigSubForm';
