import { Suggest } from '@yandex-data-ui/common';
import { EMPTY_VALUE, FieldLayout2, hasIncluded, useDismounted } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback, useEffect, useMemo, useRef, useState } from 'react';
import { firstValueFrom, takeUntil } from 'rxjs';

import { DeployUnit } from '../../../../../../models/ui';
import { useConfig, ypApi } from '../../../../../../services';
import { NetworkIdHint } from '../../../hints';

import classes from './NetworkSuggest.module.css';

interface NetworkSuggestProps {
   name: string;
   readonly: boolean;
   disabled: boolean;
}

export const NetworkSuggest: React.FC<NetworkSuggestProps> = React.memo(({ name, readonly, disabled }) => {
   const form = useFormikContext();
   const { networkId } = useMemo(() => (form.values as DeployUnit).networkDefaults, [form]);

   const [query, setQuery] = useState(networkId ?? '');

   const dismounted = useDismounted();

   const config = useConfig()!;
   const items = useRef<string[]>();

   const getItems = useCallback(
      (suggestQuery: string) => {
         if (!items.current) {
            return firstValueFrom(ypApi.getAvailableNetworkProjects(config.user.login).pipe(takeUntil(dismounted)))
               .catch(() => [] as string[])
               .then(networkProjects => {
                  items.current = networkProjects;
                  return items.current.filter(item => hasIncluded(suggestQuery, item));
               });
         }
         return items.current.filter(item => hasIncluded(suggestQuery, item));
      },
      [dismounted, config.user.login],
   );

   useEffect(() => {
      setQuery(networkId ?? '');
   }, [networkId]);

   const handleChange = useCallback(
      (value: string | null) => {
         form.setFieldValue(name, value);
      },
      [form, name],
   );

   const handleBlur = useCallback(() => {
      form.setFieldTouched(name, true, true);
   }, [form, name]);

   const handleQueryChange = useCallback(
      (q: string) => {
         setQuery(q);
         handleChange(q);
      },
      [handleChange],
   );

   const handleItemClick = useCallback(
      (item: string) => {
         setQuery(item);
         handleChange(item);
      },
      [handleChange],
   );

   return (
      <FieldLayout2
         name={name}
         label={'Network ID'}
         required={true}
         disabled={disabled}
         readonly={readonly}
         hint={<NetworkIdHint />}
         readonlyDots={readonly}
      >
         {readonly ? (
            <div className={classes.readonly}>{networkId ?? EMPTY_VALUE}</div>
         ) : (
            <Suggest<string>
               getItems={getItems}
               renderItem={item => <div className={classes.item}>{item}</div>}
               onItemClick={handleItemClick}
               onUpdate={handleQueryChange}
               placeholder={'Type to search...'}
               debounce={250}
               disabled={disabled || readonly}
               onBlur={handleBlur}
               text={query}
               showItemsOnNoText={true}
               getItemsOnMount={true}
               virtualized={true}
               itemHeight={35}
            />
         )}
      </FieldLayout2>
   );
});

NetworkSuggest.displayName = 'NetworkSuggest';
