import { List, ListProps } from '@yandex-cloud/uikit';

import { classNames, deepClone, EnumField2, EnumOption, IListOption } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import { CheckBox } from 'lego-on-react';
import React, { useCallback, useEffect, useMemo } from 'react';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faAngleDoubleRight, faRandom } from '@fortawesome/pro-regular-svg-icons';
import { PerLocationSettings, DeployUnitType, PerLocationStrategy } from '../../../../../models/ui';

import { useConfig } from '../../../../../services';
import { noop } from '../../../../../utils';
import { EnabledSwitcherField } from '../../../../forms';

import classes from './PerLocationSettingsSubForm.module.css';

type LocationItemProps = {
   index: number;
   name: string;
   approval: boolean;
   toggleNeedApproval?(): void;
   readonly: boolean;
   strategy: PerLocationStrategy;
};

const LocationItem: React.FC<LocationItemProps> = React.memo(
   ({ index, name, approval, toggleNeedApproval, readonly, strategy }) => (
      <div
         className={classNames(classes.locationItem, {
            [classes.locationRow]: readonly,
            [classes.parallelStrategyLine]: strategy === PerLocationStrategy.Parallel,
         })}
      >
         {strategy === PerLocationStrategy.Sequential && <div>{index + 1}</div>}
         <div>{name}</div>
         <div>
            {readonly ? (
               approval ? (
                  'enabled'
               ) : (
                  'disabled'
               )
            ) : (
               <CheckBox
                  theme={'normal'}
                  size={'s'}
                  view={'default'}
                  tone={'grey'}
                  checked={approval}
                  disabled={readonly}
                  onChange={toggleNeedApproval ?? noop}
               />
            )}
         </div>
      </div>
   ),
);

const strategyOptions: EnumOption[] = [
   {
      value: PerLocationStrategy.Parallel,
      title: (
         <>
            Parallel
            <span className={classes.strategyIcon}>
               <FontAwesomeIcon icon={faRandom} />
            </span>
         </>
      ),
   },
   {
      value: PerLocationStrategy.Sequential,
      title: (
         <>
            Sequential
            <span className={classes.strategyIcon}>
               <FontAwesomeIcon icon={faAngleDoubleRight} />
            </span>
         </>
      ),
   },
];

interface PerLocationSettingsProps {
   name: string;
   deployUnitType: DeployUnitType;
   readonly: boolean;
   disabled: boolean;
   locations: string[];
}

export const PerLocationSettingsSubForm: React.FC<PerLocationSettingsProps> = React.memo(
   ({ locations, readonly, disabled, name, deployUnitType }) => {
      const form = useFormikContext();
      const { value } = form.getFieldMeta<PerLocationSettings>(name);
      const { needApproval, locationOrder, strategy, isCustom } = value;

      const toggleNeedApproval = useCallback(
         (location: string) => {
            const newNeedApproval = new Set(Array.from(needApproval.values()));
            if (needApproval.has(location)) {
               newNeedApproval.delete(location);
            } else {
               newNeedApproval.add(location);
            }
            form.setFieldValue(`${name}.needApproval`, deepClone(newNeedApproval));
         },
         [needApproval, name, form],
      );

      const { clusters } = useConfig()!;
      const locationItems = useMemo(() => {
         const clusterNames = clusters.reduce((names, cluster) => {
            names[cluster.value] = {
               name: cluster.name,
               value: cluster.value,
            };
            return names;
         }, {} as Record<string, IListOption>);
         return locationOrder.map(e => clusterNames[e]);
      }, [clusters, locationOrder]);

      const onSortEnd: Exclude<ListProps<IListOption>['onSortEnd'], undefined> = useCallback(
         ({ oldIndex, newIndex }) => {
            if (oldIndex === newIndex) {
               return;
            }
            form.setFieldValue(`${name}.locationOrder`, [
               ...List.moveListElement(locationOrder.slice(), oldIndex, newIndex),
            ]);
         },
         [locationOrder, name, form],
      );

      useEffect(() => {
         const locationsSet = new Set(locations);
         const newLocationOrder: string[] = [];
         for (const location of locationOrder) {
            if (locationsSet.has(location)) {
               newLocationOrder.push(location);
            }
         }
         const locationOrderSet = new Set(newLocationOrder);
         for (const location of locations) {
            if (!locationOrderSet.has(location)) {
               newLocationOrder.push(location);
            }
         }

         if (isCustom) {
            form.setFieldValue(`${name}.locationOrder`, newLocationOrder);
         }

         // form исключена из зависимостей, чтобы предотвратить бесконечные вызовы
         // eslint-disable-next-line react-hooks/exhaustive-deps
      }, [locations, name, strategy]);

      if (deployUnitType === DeployUnitType.MultiCluster) {
         return null;
      }

      const isSequential = value.strategy === PerLocationStrategy.Sequential;

      return (
         <div className={readonly ? undefined : classes.column}>
            <EnabledSwitcherField
               name={`${name}.isCustom`}
               readonly={readonly}
               readonlyDots={readonly}
               disabled={disabled}
               label={'Custom deploying strategy'}
            />
            {value.isCustom && (
               <>
                  <EnumField2
                     name={`${name}.strategy`}
                     label={'Strategy'}
                     disabled={disabled}
                     readonly={readonly}
                     readonlyDots={readonly}
                     controlProps={{
                        options: strategyOptions,
                     }}
                  />

                  {locationItems.length > 0 && (
                     <div className={classNames({ [classes.readonly]: readonly })}>
                        <div
                           className={classNames(classes.orderHeader, {
                              [classes.parallelStrategyLine]: !isSequential,
                           })}
                        >
                           {isSequential && <div>№</div>}
                           <div>Location</div>
                           <div>Need approval</div>
                        </div>
                        <div className={classNames(classes.locationList, { [classes.parallelList]: !isSequential })}>
                           {readonly ? (
                              locationItems.map((item, index) => (
                                 <LocationItem
                                    name={item.name}
                                    index={index}
                                    readonly={readonly}
                                    approval={needApproval.has(item.value)}
                                    strategy={value.strategy}
                                 />
                              ))
                           ) : (
                              <List
                                 items={locationItems}
                                 filterable={false}
                                 sortable={isSequential}
                                 itemHeight={28}
                                 virtualized={false}
                                 itemClassName={classes.locationRow}
                                 onSortEnd={onSortEnd}
                                 sortHandleAlign={'right'}
                                 renderItem={(item, _, index) => (
                                    <LocationItem
                                       name={item.name}
                                       index={index}
                                       readonly={readonly}
                                       approval={needApproval.has(item.value)}
                                       toggleNeedApproval={() => toggleNeedApproval(item.value)}
                                       strategy={value.strategy}
                                    />
                                 )}
                              />
                           )}
                        </div>
                     </div>
                  )}
               </>
            )}
         </div>
      );
   },
);

PerLocationSettingsSubForm.displayName = 'PerLocationSettingsSubForm';
