import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import {
   classNames,
   EnumField2,
   FieldLayout2,
   FormButton,
   InputField2,
   RemoveButton,
} from '@yandex-infracloud-ui/libs';
import { getIn, useFormikContext } from 'formik';
import React, { useCallback } from 'react';

import { TvmClient, TvmClientMode, TvmClientModeOptions } from '../../../../../models/ui';
import { SecretSubForm, useSecretsContext } from '../../../../../modules/secrets';
import { FormSubSection } from '../../../../forms';
import { TvmSecretHint } from './hints';

import classes from './TvmClientSubForm.module.css';
import { TvmDestinationRow } from './TvmDestinationRow';

interface Props {
   canRemove: boolean;
   disabled: boolean;
   index: number;
   name: string;
   readonly: boolean;

   onRemove(i: number): void;
}

export const TvmClientSubForm: React.FC<Props> = React.memo(
   ({ name, index, readonly, disabled, canRemove, onRemove }) => {
      const { stageId, duId, reloadUsages } = useSecretsContext();

      const form = useFormikContext();
      const value: TvmClient = getIn(form.values, name);

      const handleRemoveClient = useCallback(() => onRemove!(index!), [index, onRemove]);

      const handleAddDestination = useCallback(() => {
         const newDestinations = [...(value.destinations ?? []), { app: null, alias: null }];

         form.setFieldValue(`${name}.destinations`, newDestinations);
      }, [form, name, value.destinations]);

      const handleDestinationRemove = useCallback(
         (i: number) => {
            if (value.destinations?.[i]) {
               const newDestinations = [...value.destinations];
               newDestinations.splice(i, 1);

               form.setFieldValue(`${name}.destinations`, newDestinations);
            }
         },
         [form, name, value.destinations],
      );

      const isEditable = !(disabled || readonly);

      const useDestinations = value.mode !== TvmClientMode.CheckOnly;
      const useSecret = value.mode !== TvmClientMode.CheckOnly;

      return (
         <div data-e2e={`TvmClient:${index + 1}`}>
            <FormSubSection>
               {canRemove && !disabled && !readonly ? (
                  <div className={classes.removeClient} data-e2e={'TvmClient:RemoveClientButton'}>
                     <RemoveButton onClick={handleRemoveClient} size={'lg'} />
                  </div>
               ) : null}

               <FieldLayout2 name={name} label={`Client ${index + 1}`} bigLabel={true}>
                  <div className={classNames(classes.clientMode, classes.hideLabel)} data-e2e={`TvmClient:Mode`}>
                     <EnumField2
                        name={`${name}.mode`}
                        controlProps={{ options: TvmClientModeOptions }}
                        readonly={readonly}
                        disabled={disabled}
                     />
                  </div>

                  <table className={classes.table} data-e2e={`TvmClient:Table`}>
                     <thead>
                        <tr>
                           <th aria-label={'Label'} />
                           <th>TVM app</th>
                           <th>Alias</th>
                           <th aria-label={'spacer'} />
                        </tr>
                     </thead>
                     <tbody>
                        <tr>
                           <td>
                              <div className={readonly ? classes.source : undefined}>Source</div>
                           </td>
                           <td>
                              <InputField2
                                 name={`${name}.source.app`}
                                 label={null}
                                 controlProps={{ controlProps: { min: 0 }, type: 'number' }}
                                 readonly={readonly}
                                 disabled={disabled}
                              />
                           </td>
                           <td>
                              <InputField2
                                 name={`${name}.source.alias`}
                                 label={null}
                                 placeholder={value.source?.app?.toString() ?? ''}
                                 readonly={readonly}
                                 disabled={disabled}
                              />
                           </td>
                           <td />
                        </tr>

                        {useDestinations
                           ? value.destinations?.map((_, i) => (
                                <TvmDestinationRow
                                   // eslint-disable-next-line react/no-array-index-key
                                   key={i}
                                   canRemove={(value.destinations?.length ?? 0) > 1}
                                   disabled={disabled}
                                   index={i}
                                   name={`${name}.destinations[${i}]`}
                                   onRemove={handleDestinationRemove}
                                   readonly={readonly}
                                   required={!readonly && value.source.app !== null && i === 0}
                                />
                             ))
                           : null}
                     </tbody>
                  </table>

                  {isEditable && useDestinations ? (
                     <div className={classes.addDestination}>
                        <FormButton
                           onClick={handleAddDestination}
                           skipLeftSpace={true}
                           icon={faPlus}
                           dataE2e={'TvmClient:AddDestinationButton'}
                        >
                           Add destination
                        </FormButton>
                     </div>
                  ) : null}

                  {useSecret ? (
                     <div className={classes.secret} data-e2e={'TvmClient:Secret'}>
                        <FieldLayout2
                           name={`${name}.secret`}
                           label={'Secret'}
                           hint={<TvmSecretHint />}
                           required={value.source.app !== null}
                        >
                           <SecretSubForm
                              duId={duId}
                              name={`${name}.secret`}
                              stageId={stageId}
                              layout={'vertical'}
                              disabled={disabled}
                              readonly={readonly}
                              onChange={reloadUsages}
                           />
                        </FieldLayout2>
                     </div>
                  ) : null}
               </FieldLayout2>
            </FormSubSection>
         </div>
      );
   },
);

TvmClientSubForm.displayName = 'TvmClientSubForm';
