import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { FieldLayout2, FormAdvancedSection, FormButton, InputField2, isEmpty } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback, useMemo, useState } from 'react';

import { DEFAULT_TVM_CLIENT_PORT } from '../../../../../models';
import {
   Box,
   DeployUnit,
   DeployUnitFormParams,
   DeployUnitTvm,
   getDefaultTvmClient,
   SidecarName,
   TvmClient,
} from '../../../../../models/ui';
import { useSecretsContext } from '../../../../../modules/secrets';
import { pluralNumber } from '../../../../../utils';
import { EnabledSwitcher } from '../../../../forms';
import { SubForm } from '../../../../huge-form';
import { SidecarResourcePlate } from '../../../../stage-huge-form/components';

import { BlackboxSelectFieldField } from '../../fields/BlackboxSelectField/BlackboxSelectField';
import { TvmBlackboxHint, TvmClientPortHint, TvmHint } from './hints';
import { TvmClientSubForm } from './TvmClientSubForm';
import classes from './TvmSubForm.module.css';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
   form?: SubForm<DeployUnitFormParams, DeployUnit, Box>;
}

export const TvmSubForm: React.FC<Props> = React.memo(({ readonly, disabled, name, form: subForm }) => {
   const { reloadUsages } = useSecretsContext();

   const form = useFormikContext();
   const { value } = form.getFieldMeta<DeployUnitTvm>(name);
   const [opened, setOpened] = useState(false);

   const countClients = useMemo(() => {
      const clients = value.clients?.filter(client => !isEmpty(client?.source?.app)) ?? [];

      return clients.length;
   }, [value.clients]);

   const handleEnabledChange = useCallback(
      (v: boolean) => {
         form.setFieldValue(`${name}.enabled`, v);
         setOpened(v);
      },
      [form, name],
   );

   const handleAddClient = useCallback(() => {
      const newClients = [...value.clients, getDefaultTvmClient()];

      form.setFieldValue(`${name}.clients`, newClients);
   }, [form, name, value.clients]);

   const handleRemoveClient = useCallback(
      (i: number) => {
         const newClients = [...value.clients];
         newClients.splice(i, 1);

         form.setFieldTouched(`${name}.clients[${i}]`, false);
         form.setFieldValue(`${name}.clients`, newClients);

         reloadUsages();
      },
      [form, name, reloadUsages, value.clients],
   );

   return (
      <FieldLayout2
         name={name}
         label={'TVM'}
         hint={<TvmHint />}
         hideErrors={true}
         readonly={readonly}
         readonlyDots={readonly && !value.enabled}
         bigLabel={!readonly || value.enabled}
      >
         <FormAdvancedSection
            header={
               <div data-e2e={'Tvm:EnabledSwitcher'}>
                  <EnabledSwitcher
                     value={value.enabled}
                     name={`${name}.enabled`}
                     onChange={handleEnabledChange}
                     readonly={readonly}
                     disabled={disabled}
                  />
               </div>
            }
            hasValue={countClients > 0}
            title={'TVM settings'}
            toggleText={pluralNumber(countClients, 'client', 'clients')}
            openedByDefault={opened || value.enabled}
            hideToggleSwitch={!value.enabled || readonly}
         >
            {value.enabled ? (
               <div className={classes.warning} data-e2e={'Tvm:Warning'}>
                  <SidecarResourcePlate sidecar={SidecarName.TVM} deployUnit={subForm?.formParams?.initialId ?? ''} />
               </div>
            ) : null}

            <div className={classes.portBlackboxRow}>
               <InputField2
                  name={`${name}.clientPort`}
                  label={'Client port'}
                  controlProps={{ controlProps: { min: 0 }, type: 'number', cls: classes.port } as any}
                  placeholder={DEFAULT_TVM_CLIENT_PORT} // TODO: constant
                  readonly={readonly}
                  disabled={disabled || !value.enabled}
                  hint={<TvmClientPortHint />}
               />

               <BlackboxSelectFieldField
                  name={`${name}.blackbox`}
                  label={'Blackbox'}
                  readonly={readonly}
                  disabled={disabled || !value.enabled}
                  hint={<TvmBlackboxHint />}
               />
            </div>

            <div className={classes.clients}>
               {value.clients.map((client: TvmClient, i: number) => (
                  <TvmClientSubForm
                     // eslint-disable-next-line react/no-array-index-key
                     key={i}
                     canRemove={value.clients.length > 1}
                     disabled={disabled || !value.enabled}
                     index={i}
                     name={`${name}.clients[${i}]`}
                     onRemove={handleRemoveClient}
                     readonly={readonly}
                  />
               ))}
            </div>

            {disabled || !value.enabled || readonly ? null : (
               <FormButton onClick={handleAddClient} skipLeftSpace={true} icon={faPlus} dataE2e={'Tvm:AddClientButton'}>
                  Add client
               </FormButton>
            )}
         </FormAdvancedSection>
      </FieldLayout2>
   );
});

TvmSubForm.displayName = 'TvmSubForm';
