import { faUndo } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button, ControlGroupOption, RadioButton } from '@yandex-cloud/uikit';
import {
   EMPTY_VALUE,
   EnumField2,
   EnumOption,
   FieldLayout2,
   FormCard,
   InputField2,
   RemoveButton,
} from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback, useState } from 'react';

import { DEFAULT_OS_LIST } from '../../../../../models';
import { customBaseLayerLabel, DiskLayer, DiskLayerType, LayerSourceFileStoragePolicy } from '../../../../../models/ui';
import { SandboxUrlField } from '../../../_common/SanboxUrlField/SandboxUrlField';
import { DiskBaseLayerHint, LayerSourceFileStoragePolicyHint } from '../../hints';

import classes from './DiskLayerCard.module.css';

const defaultTypeOptions: ControlGroupOption[] = [
   {
      value: customBaseLayerLabel,
      content: customBaseLayerLabel,
   },
   ...DEFAULT_OS_LIST.map(v => ({ value: v.value, content: v.title })),
];

const layerSourceFileStoragePolicyOptions: EnumOption[] = [
   {
      value: LayerSourceFileStoragePolicy.None,
      title: 'Default', // DEPLOY-5549
   },
   {
      value: LayerSourceFileStoragePolicy.Keep,
      title: 'Keep',
   },
   {
      value: LayerSourceFileStoragePolicy.Remove,
      title: 'Remove',
   },
];

interface Props {
   name: string;
   readonly: boolean;
   disabled: boolean;
   index: number;

   onRemove: (i: number) => void;

   onRestore(i: number): void;
}

export const DiskLayerCard: React.FC<Props> = React.memo(({ name, readonly, disabled, index, onRemove, onRestore }) => {
   const form = useFormikContext();
   const { value } = form.getFieldMeta<DiskLayer>(name);

   const [baseType, setBaseType] = useState<string>(
      DEFAULT_OS_LIST.find(v => v.value === value.url)?.value ?? customBaseLayerLabel,
   );

   const handleChangeType = useCallback(
      (type: string) => {
         setBaseType(type);

         if (type !== customBaseLayerLabel) {
            if (type !== value.url) {
               form.setFieldValue(`${name}.url`, type);
            }
         }
      },
      [form, name, value, setBaseType],
   );

   const handleRemoveLayer = useCallback(() => onRemove(index), [index, onRemove]);
   const handleRestoreLayer = useCallback(() => onRestore(index), [index, onRestore]);

   return (
      <div data-test={`Layer:${index + 1}`}>
         <FormCard
            title={value.id || EMPTY_VALUE}
            className={value.removed ? classes.removed : undefined}
            buttons={
               !readonly && !disabled ? (
                  <>
                     {value.removed ? (
                        <div data-e2e={'Layer:Restore'}>
                           <Button view={'flat'} title={'Restore'} onClick={handleRestoreLayer}>
                              <FontAwesomeIcon icon={faUndo} />
                           </Button>
                        </div>
                     ) : (
                        <div data-e2e={'Layer:Remove'}>
                           <RemoveButton onClick={handleRemoveLayer} size={'lg'} />
                        </div>
                     )}
                  </>
               ) : null
            }
         >
            {value.removed === true ? (
               <FieldLayout2 name={`${name}.removed`} label={null}>
                  removed
               </FieldLayout2>
            ) : (
               <>
                  {value.type === DiskLayerType.Unknown && (
                     <div className={classes.unknown}>Unknown (for UI) layer type</div>
                  )}

                  {value.type === DiskLayerType.Url && (
                     <div className={classes.inputWidth}>
                        <InputField2
                           name={`${name}.id`}
                           label={'ID'}
                           readonly={readonly}
                           readonlyDots={readonly}
                           disabled={disabled}
                           required={true}
                        />

                        {!readonly ? (
                           <FieldLayout2
                              name={`${name}.layerType`}
                              label={'Layer type'}
                              hint={<DiskBaseLayerHint />}
                              readonly={readonly}
                              readonlyDots={readonly}
                           >
                              <div className={classes.layerType}>
                                 <RadioButton
                                    options={defaultTypeOptions}
                                    value={baseType}
                                    onUpdate={handleChangeType}
                                 />
                              </div>
                           </FieldLayout2>
                        ) : null}

                        <SandboxUrlField
                           name={`${name}.url`}
                           label={'URL'}
                           readonly={readonly}
                           readonlyDots={readonly}
                           disabled={disabled || baseType !== customBaseLayerLabel}
                           required={true}
                        />

                        <InputField2
                           name={`${name}.checksum`}
                           label={'Checksum'}
                           readonly={readonly}
                           readonlyDots={readonly}
                           disabled={disabled}
                           required={true}
                        />

                        <EnumField2
                           name={`${name}.layerSourceFileStoragePolicy`}
                           label={'Layer storage policy'}
                           readonly={readonly}
                           readonlyDots={readonly}
                           disabled={disabled}
                           controlProps={{ options: layerSourceFileStoragePolicyOptions }}
                           hint={<LayerSourceFileStoragePolicyHint />}
                        />
                     </div>
                  )}
               </>
            )}
         </FormCard>
      </div>
   );
});

DiskLayerCard.displayName = 'DiskLayerCard';
