import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { EMPTY_VALUE, FieldLayout2, FormButton } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback } from 'react';

import { EMPTY_CHECKSUM } from '../../../../../models';
import { DiskLayer, DiskLayerType, LayerSourceFileStoragePolicy } from '../../../../../models/ui';
import { getHexRef } from '../../../../../utils';

import { DiskLayersHint } from '../../hints';
import { DiskLayerCard } from './DiskLayerCard';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
}

export const DiskLayersSubForm: React.FC<Props> = React.memo(({ disabled, name, readonly }) => {
   const form = useFormikContext();
   const { value } = form.getFieldMeta<DiskLayer[]>(name);

   const handleAddLayer = useCallback(() => {
      const newLayer: DiskLayer = {
         id: '',
         type: DiskLayerType.Url,
         url: '',
         checksum: EMPTY_CHECKSUM,
         _ref: getHexRef(),
         layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
      };

      form.setFieldValue(name, [...value, newLayer]);
   }, [form, name, value]);

   const handleRemoveLayer = useCallback(
      (i: number) => {
         const newValue = [...value].map((v, index) =>
            index === i
               ? {
                    ...v,
                    removed: true,
                 }
               : v,
         );

         form.setFieldTouched(`${name}[${i}].removed`, true);
         form.setFieldValue(name, newValue);
      },
      [form, name, value],
   );

   const handleRestoreLayer = useCallback(
      (i: number) => {
         const newValue = [...value].map((v, index) =>
            index === i
               ? {
                    ...v,
                    removed: false,
                 }
               : v,
         );

         form.setFieldTouched(`${name}[${i}].removed`, false);
         form.setFieldValue(name, newValue);
      },
      [form, name, value],
   );

   return (
      <FieldLayout2 name={name} label={'Layers'} hint={<DiskLayersHint />} hideErrors={true} bigLabel={true}>
         <div data-test={'Disk:Layers'}>
            {value.length === 0 ? (
               <>{readonly ? <div>{EMPTY_VALUE}</div> : null}</>
            ) : (
               value.map((_, i) => (
                  <DiskLayerCard
                     name={`${name}[${i}]`}
                     // eslint-disable-next-line react/no-array-index-key
                     key={i}
                     index={i}
                     disabled={disabled}
                     readonly={readonly}
                     onRemove={handleRemoveLayer}
                     onRestore={handleRestoreLayer}
                  />
               ))
            )}

            {disabled || readonly ? null : (
               <FormButton icon={faPlus} onClick={handleAddLayer} skipLeftSpace={true} dataE2e={'Layers:AddLayer'}>
                  Add layer
               </FormButton>
            )}
         </div>
      </FieldLayout2>
   );
});

DiskLayersSubForm.displayName = 'DiskLayersSubForm';
